'' This object generates a 640x480 VGA signal which contains 40 columns x 25
'' rows of 8x8 characters with double size pixels. Each character can have a
'' unique forground colour and there is a whole screen background colour.
''
'' You must provide buffers for the screen, colours, font, CRTC and sync.
'' Once started, all interfacing is done via memory. To this object, all buffers
'' are read-only, with the exception of the sync indicator which gets written with
'' -1. You may freely write all buffers to affect screen appearance. Have fun!
''

CON
{
' 800 x 600 @ 75Hz settings: 100 x 50 characters
	hp	= 800		' horizontal pixels
	vp	= 600		' vertical pixels
	hf	= 40+80		' horizontal front porch pixels
	hs	= 128		' horizontal sync pixels
	hb	= 88+80		' horizontal back porch pixels
	vf	= 1+100		' vertical front porch lines
	vs	= 4		' vertical sync lines
	vb	= 23+100	' vertical back porch lines
	hn	= 0		' horizontal normal sync state (0|1)
	vn	= 0		' vertical normal sync state (0|1)
	pr	= 50		' pixel rate in MHz at 80MHz system clock (5MHz granularity)
}
' 640 x 480 @ 75Hz settings: 40 x 25 characters (doubled pixels)
	hp	= 640		' horizontal pixels
	vp	= 480		' vertical pixels
	hf	= 24+48		' horizontal front porch pixels
	hs	= 40		' horizontal sync pixels
	hb	= 128+48	' horizontal back porch pixels
	vf	= 20+40		' vertical front porch lines
	vs	= 3		' vertical sync lines
	vb	= 17+40		' vertical back porch lines
	hn	= 1		' horizontal normal sync state (0|1)
	vn	= 1		' vertical normal sync state (0|1)
	pr	= 35		' pixel rate in MHz at 80MHz system clock (5MHz granularity)

' columns and rows
	cols = 40
	rows = 25


VAR long cog[2]

PUB start(basepin, params) : okay | i, j

'' Start VGA driver - starts two COGs
'' returns false if two COGs not available
''
''	basepin = VGA starting pin (0, 8, 16, 24, etc.)
''
''	params[0] = Pointer to 40x108 bytes containing ASCII codes or pixel data for
''              each of the 40x25 screen characters or 160x108 2x2 pixels. Screen
''              memory is arranged left-to-right, top-to-bottom.
''
''	params[1] = Pointer to 1,024 byte which define the foreground colours
''		colours for each character in text mode. The lower 4 bits define an
''              index into a colour palette of 16 entries.
''
''	params[2] = Pointer to 1,024 byte which define the upper 128 characters
''		if the corresponding bits of mode are set to 1.
''
''	params[3] = Pointer to 18 bytes which control the display
''                  00 - horizontal total characters
''                  01 - horizontal displayed characters per line
''                  02 - horizontal synch position
''                  03 - horizontal synch width in characters
''                  04 - vertical total lines
''                  05 - vertical total adjust (scan lines)
''                  06 - vertical displayed rows
''                  07 - vertical synch position (character rows)
''                  08 - interlace mode
''                  09 - maximum scan line address
''                  0a - cursor start (scan line)
''                  0b - cursor end (scan line)
''                  0c - start address (MSB)
''                  0d - start address (LSB)
''                  0e - cursor address (MSB) (RW)
''                  0f - cursor address (LSB) (RW)
''                  10 - light pen (MSB) (RO)
''                  11 - light pen (LSB) (RO)
''
''	params[4] = Pointer to the portff long
''
''	params[7] = Pointer to long which gets written with -1 upon each screen
''		refresh. May be used to time writes/scrolls, so that chopiness
''		can be avoided. You must clear it each time if you want to see
''		it re-trigger.

	' if driver is already running, stop it
	stop

	' implant pin settings
	vcfg_2colour := $200000ff + ((basepin & %111000) << 6)
	vcfg_4colour := $300000ff + ((basepin & %111000) << 6)
	i := $FF << (basepin & %011000)
	j := basepin & %100000 == 0
	reg_dira := i & j
	reg_dirb := i & !j

	' implant CNT value to sync COGs to
	sync_cnt := cnt + $10000

	' implant pointers
	longmove(@screen_base, params, 7)
	font_base := @font

	' implant unique settings and launch first COG
	vf_lines.byte := vf
	vb_lines.byte := vb
	font_half := 1
	cog[1] := cognew(@entry, long[params][7]) + 1

	' allow time for first COG to launch
	waitcnt($2000 + cnt)

	' differentiate settings and launch second COG
	vf_lines.byte := vf+8
	vb_lines.byte := vb-8
	font_half := 0
	cog[0] := cognew(@entry, long[params][7]) + 1

	' if both COGs launched, return true
	if cog[0] and cog[1]
		return 1

	' else, stop any launched COG and return false
	stop


PUB stop | i

'' Stop VGA driver - frees two COGs

	repeat i from 0 to 1
		if cog[i]
			cogstop(cog[i]~ - 1)


PUB write_chargen(offs, src, size) | addr, i
'' Write the character generator
	repeat i from offs to offs + size - 1
		addr := (i & $003) + ((i & $ff0) >> 2) + ((i & $00c) * 256)
		byte[@font][addr] := byte[src++] >< 8

CON
	hv_inactive = (hn << 1 + vn) * $01010101		'H,V inactive states


DAT

'*****************************************************
'* Assembly language VGA high-resolution text driver *
'*****************************************************

' This program runs concurrently in two different COGs.
'
' Each COG's program has different values implanted for front-porch lines and
' back-porch lines which surround the vertical sync pulse lines. This allows
' timed interleaving of their active display signals during the visible portion
' of the field scan. Also, they are differentiated so that one COG displays
' even four-line groups while the other COG displays odd four-line groups.
'
' These COGs are launched in the PUB 'start' and are programmed to synchronize
' their PLL-driven video circuits so that they can alternately prepare sets of
' four scan lines and then display them. The COG-to-COG switchover is seamless
' due to two things: exact synchronization of the two video circuits and the
' fact that all COGs' driven output states get OR'd together, allowing one COG
' to output lows during its preparatory state while the other COG effectively
' drives the pins to create the visible and sync portions of its scan lines.
' During non-visible scan lines, both COGs output together in unison.
'

			org	0				' set origin to $000 for start of program
entry
			' Init I/O registers and sync COGs' video circuits
			mov	dira, reg_dira			' set pin directions
			mov	dirb, reg_dirb
			movi	frqa, #(pr / 5) << 2		' set pixel rate
			mov	vcfg, vcfg_2colour		' set video configuration
			mov	vscl, #1			' set video to reload on every pixel
			waitcnt sync_cnt, onems			' wait for start value in cnt, add ~1ms
			movi	ctra, #%00001_110		' COGs in sync! enable PLLs now - NCOs locked!
			waitcnt sync_cnt, #0			' wait ~1ms for PLLs to stabilize - PLLs locked!
			mov	vscl, #100			' insure initial WAITVIDs lock cleanly

' Main loop, display field - each COG alternately builds and displays four scan lines

vsync			mov	x, #vs				' do vertical sync lines
			call	#blank_vsync

			rdbyte	portff, portff_ptr		' get current port $FF value
			test	portff, #%00000100	wz	' background enabled?
	if_nz		mov	bgcolour, bgcolour_1		' yes, use background colour 1
	if_nz		jmp	#:newbg
			mov	bgcolour, bgcolour_0		' no, use background colour 0
			test	portff, #%01000000	wc	' except if bits 6
			test	portff, #%10000000	wz	' or 7 are set
	if_c_and_z	mov	bgcolour, bgcolour_01		' select background color 01
	if_nc_and_nz	mov	bgcolour, bgcolour_02		' 02
	if_c_and_nz	mov	bgcolour, bgcolour_03		' or 03
:newbg
			rdlong	screen_addr, start_ptr		' get CRTC register $0c/$0d (start address)

			mov	z, #$01				' get CRTC register $01 (horizontal displayed)
			add	z, crtc_ptr
			rdbyte	horz_disp, z		wz
		if_z	mov	horz_disp, #40
			cmp	horz_disp, horz_setup	wz
		if_z	jmp	#:scancode_ok

			mov	colcount, horz_disp
			movd	:dest1, #scancode
			movd	:dest2, #scancode+1
			mov	i_waitvid, i_waitvid0
			mov	i_shr, i_shr0

			mov	xhf, horz_disp			' adjust horizontal front porch
			sub	xhf, #40			' - 40
			shl	xhf, #3				' * 8 pixels
			mov	xhb, xhf
			add	xhf, #hf			' + value for 40 columns
			add	xhb, #hb			' + value for 40 columns
	
:dest1			mov	0-0, i_waitvid
:dest2			mov	0-0, i_shr
			add	:dest1, d1
			add	:dest2, d1
			add	i_waitvid, d0s0
			add	i_shr, d0
			djnz	colcount, #:dest1
			mov	x, #scancode
			add	x, horz_disp
			add	x, horz_disp
			movd	:dest3, x
			mov	horz_setup, horz_disp		' setup done for horizontal displayed
:dest3			mov	0-0, i_jmp_scanret
:scancode_ok
			mov	screen_ptr, screen_base		' reset screen pointer to upper-left character
			mov	colour_ptr, #0			' reset colour pointer to 0

vb_lines		mov	x,#vb				' do vertical back porch lines (# set at runtime)
			call	#blank_vsync

			mov	z, #6				' vertical displayed
			add	z, crtc_ptr
			rdbyte	rowcount, z
			add	z, #3
			rdbyte	scanlines, z		wz	' 1 pixel row per scan row
			add	scanlines, #1
			cmp	scanlines, #2		wz	' 2 pixel rows per scan row (default for graphics)
		if_z	shr	rowcount, #2			' !!!!!!!! doesn't work - why ??????????
'			mov	rowcount, #rows

			' Build four scan lines into scanbuff
nextrow
			test	portff, #%00100000	wz	' graphics mode?
		if_z	jmp	#textmode

graphicsmode
			mov	vcfg, vcfg_4colour
			mov	vscl_chr, vscl_gfx

			movd	:pixa, #scanbuff-1		' reset scanbuff address (pre-decremented)
			movd	:cola, #colorbuff-1		' reset colorbuff address (pre-decremented)

			mov	colour, grcolour
			or	colour, bgcolour

			mov	gfxptr0, screen_ptr		' first row

			cmp	scanlines, #2		wz	' change on every other scanline?
		if_z	mov	x, horz_disp
		if_nz	mov	x, #0
			cmp	font_half, #1		wz	' 2nd cog?
		if_z	mov	z, horz_disp			' 2nd cog displays the other half of the scan rows
		if_nz	mov	z, #0
			shl	z, #1
			add	gfxptr0, z
			mov	gfxptr1, gfxptr0
			add	gfxptr1, x

			mov	y, #4				' must build scanbuff in four sections because
:quarterrow		mov	vscl, vscl_line2x		' ..pixel counter is limited to twelve bits
			waitvid underscore, #0			' output lows to let other COG drive VGA pins
			mov	x, horz_disp			' ..for 2 scan lines, ready for a quarter row
			shr	x, #2

:column			rdbyte	z, gfxptr0			' get character from screen memory
			rev	z, #24
			mov	w, z
			ror	w, #8
			or	w, z
			rdbyte	z, gfxptr1
			ror	w, #8
			rev	z, #24
			or	w, z
			ror	w, #8
			or	w, z
			ror	w, #8
			add	:pixa, d0			' increment scanbuff destination addresses
			add	:cola, d0
			add	gfxptr0, #1
			add	gfxptr1, #1
:pixa			mov	scanbuff, w			' read pixel long (8*4) into scanbuff
:cola			mov	colorbuff, colour

			djnz	x, #:column			' another character in this half-row?
			djnz	y, #:quarterrow			' loop to do next quarter row, time for WAITVID

			add	screen_ptr, #4 * cols
			jmp	#display

textmode
			mov	vcfg, vcfg_2colour
			mov	vscl_chr, vscl_txt
			mov	font_ptr, font_half		' get address of appropriate font section
			shl	font_ptr, #8+2			' shl 8 for 256 characters, 2 for indexing longs
			add	font_ptr, font_base

			movd	:pixa, #scanbuff-1		' reset scanbuff address (pre-decremented)
			movd	:cola, #colorbuff-1		' reset colorbuff address (pre-decremented)
			movd	:colb, #colorbuff-1

			mov	y, #4				' must build scanbuff in four sections because
:quarterrow		mov	vscl, vscl_line2x		' ..pixel counter is limited to twelve bits
			waitvid underscore, #0			' output lows to let other COG drive VGA pins
			mov	x, horz_disp			' ..for 2 scan lines, ready for a quarter row
			shr	x, #2

:column			rdbyte	z, screen_ptr			' get character from screen memory
			add	:pixa, d0			' increment scanbuff destination addresses
			cmp	z, #$80			wc
	if_c		jmp	#:default_font
			test	z, #%01000000		wz	' 1st or 2nd half of high characters?
	if_z		test	portff, #%00010000	wc
	if_nz		test	portff, #%00001000	wc
	if_c		jmp	#:default_font			' use default font
			sub	z, #$80
			shl	z, #1				' character code in bits 8..1
			or	z, font_half			' add font half
			shl	z, #2				' code in bits 9..3, half in bit 2
			add	z, font_user			' add user font base
			jmp	#:pixa
:default_font		shl	z, #2				' character code in bits 9..2
			add	z, font_ptr			' add font section address to point to 8*4 pixels
:pixa			rdlong	scanbuff, z			' read pixel long (8*4) into scanbuff
			add	screen_ptr, #1			' increment screen memory address
			mov	z, colour_base
			and	colour_ptr, address_mask	' stay in 1K of memory
			add	z, colour_ptr
			rdbyte	z, z				' fetch colour value
			and	z, #$0f				' 4 bit memory: 16 colours
			add	z, #palette			' index into palette
			movs	:cola, z
			add	:cola, d0			' increment colorbuff destination addresses
			add	:colb, d0
			add	colour_ptr, #1			' increment colour memory address
:cola			mov	colorbuff, 0-0
:colb			or	colorbuff, bgcolour

			djnz	x, #:column			' another character in this half-row?
			djnz	y, #:quarterrow			' loop to do next quarter row, time for WAITVID

			' Insert cursor into scanbuff
			'
			' This is either broken or takes too long(?)
			'
'*************
			mov	z, #$0a
			add	z, crtc_ptr
			rdbyte	cursor_first, z			' get cursor start scan + flags ($0a)
			add	z, #1
			rdbyte	cursor_last, z			' get cursor end scan ($0b)
			add	z, #3
			rdbyte	cursor, z			' get cursor address MSB ($0e)
			add	z, #1
			and	cursor, #$3f			' mask undefined bits
			add	cursor, #$40
			shl	cursor, #8
			rdbyte	x, z				' get cursor address LSB ($0f)
			or	cursor, x			' add to MSB*256
'*************
			sub	cursor, screen_addr	wc	' cursor below screen address?
	if_c		jmp	#:nocursor			' yes, don't display it
			cmp	cursor, #cols		wc	' cursor after end of this row?
	if_nc		jmp	#:nocursor			' yes, don't display it (yet)
			add	cursor, #scanbuff		' cursor in scanbuff, add scanbuff address
			movd	:cursor, cursor
			test	cursor_first, #%00100000 wc	' get mode bit 0
			test	cursor_first, #%01000000 wz	' get mode bit 1
	if_c_and_z	jmp	#:nocursor			' $20 = cursor off
	if_nc_and_z	jmp	#:cursor			' $00 = always on
	if_c_and_nz	test	slowbit, cnt		wz	' $60 = 32 frame blink (slow blink)
	if_nc_and_nz	test	fastbit, cnt		wz	' $40 = 16 frame blink (fast blink)
:cursor	if_z		mov	0-0, longmask			' conditionally put cursor into scanbuff

:nocursor

display			' Display four scan lines from scanbuff
			mov	y, #4				' ready for four times two scan lines
scanline		mov	x, #2			wc	' clear carry and set sweep count to 2
			mov	colcount, horz_disp		' set number of columns
sweep			mov	vscl, vscl_chr			' set pixel rate for characters
			jmp	#scancode

scanret			mov	vscl, xhf			' do horizontal front porch pixels
			waitvid hvsync, #0			' #0 makes hsync inactive
			mov	vscl, #hs			' do horizontal sync pixels
			waitvid hvsync, #1			' #1 makes hsync active
			mov	vscl, xhb			' do horizontal back porch pixels
			waitvid hvsync, #0			' #0 makes hsync inactive

			test	x, #2			wc	' set carry flag
			djnz	x, #sweep			' second sweep of the same data?
			djnz	y, #scanline			' another scan line?

			add	screen_addr, horz_disp		' increment screen address

			' Next group of four scan lines
			djnz	rowcount, #nextrow		' another 4-line build/display?

			' Visible section done, do vertical sync front porch lines
			wrlong	longmask, par			' write -1 to refresh indicator
vf_lines		mov	x, #vf				' do vertical front porch lines (# set at runtime)
			call	#blank

			jmp	#vsync				' new field, loop to vsync


blank_vsync		' Subroutine - do blank lines
			xor	hvsync, vertical		' flip vertical sync bits
blank
			mov	vscl, hx			' do blank pixels
			waitvid hvsync, #0
			mov	vscl, #hf			' do horizontal front porch pixels
			waitvid hvsync, #0
			mov	vscl, #hs			' do horizontal sync pixels
			waitvid hvsync, #1
			mov	vscl, #hb			' do horizontal back porch pixels
			waitvid hvsync, #0
			djnz	x, #blank			' another line?
blank_ret
blank_vsync_ret
			ret

			' Data

palette
			long	%00000011_00000011_01010111_00000011		' dark gray
			long	%00000011_00000011_00010111_00000011		' cyan
			long	%00000011_00000011_01000011_00000011		' red
			long	%00000011_00000011_10101011_00000011		' white
			long	%00000011_00000011_01010011_00000011		' yellow
			long	%00000011_00000011_00100011_00000011		' green
			long	%00000011_00000011_10100011_00000011		' orange
			long	%00000011_00000011_11110011_00000011		' light yellow
			long	%00000011_00000011_00000111_00000011		' blue
			long	%00000011_00000011_00001111_00000011		' light blue
			long	%00000011_00000011_11000111_00000011		' pink
			long	%00000011_00000011_11001011_00000011		' purple
			long	%00000011_00000011_10101011_00000011		' light gray
			long	%00000011_00000011_00111111_00000011		' light cyan
			long	%00000011_00000011_11001111_00000011		' magenta
			long	%00000011_00000011_11111111_00000011		' bright white

colour			long	0
bgcolour		long	%00000011_00000011_00000011_00000011		' currently selected background color
bgcolour_0		long	%00000011_00000011_00000011_00000011
bgcolour_1		long	%00000011_00000011_00000011_10001011
bgcolour_01		long	%00000011_00000011_00000011_10010111
bgcolour_02		long	%00000011_00000011_00000011_01100111
bgcolour_03		long	%00000011_00000011_00000011_01010111

			'	green     blue     orange   background
grcolour		long	%00110011_00001011_11010011_00000011

screen_base		long	0				' set at runtime (7 contiguous longs)
start_ptr		long	0				' set at runtime
colour_base		long	0				' set at runtime
font_user		long	0				' set at runtime
crtc_ptr		long	0				' set at runtime
psg_ptr			long	0				' set at runtime
portff_ptr		long	0				' set at runtime

font_base		long	0				' set at runtime
font_half		long	0				' set at runtime
portff			long	$18				' bit 0:    cassette output
								' bit 1:    unused
								' bit 2:    1: background enable
								' bit 3:    0: user chargen $c0..$ff
								' bit 4:    0: user chargen $80..$bf
								' bit 5:    1: graphics mode 0: text mode
								' bit 7..6: background colour 0-3
vcfg_4colour		long	0-0				' 4 colour mode - set at runtime
vcfg_2colour		long	0-0				' 2 colour mode - set at runtime
reg_dira		long	0-0				' set at runtime
reg_dirb		long	0-0				' set at runtime
sync_cnt		long	0-0				' set at runtime
hx			long	hp				' visible pixels per scan line
vscl_line2x		long	(hp + hf + hs + hb) * 2 	' total number of pixels per 2 scan lines
vscl_chr		long	2 << 12 + 2 * 8			' vscale value for the character data
vscl_txt		long	2 << 12 + 2 * 8			' vscale value for the text data
vscl_gfx		long	4 << 12 + 4 * 4			' vscale value for the graphics data
onems			long	$fcfc				' timing value for first waitcnt
longmask		long	$ffffffff			' all bits set
slowbit			long	1 << 25				' cnt mask for slow cursor blink
fastbit			long	1 << 24				' cnt mask for fast cursor blink
underscore		long	$ffff0000			' underscore cursor pattern
hv			long	hv_inactive			' -H,-V states
vertical		long	$01010101
hvsync			long	hv_inactive ^ $02000200		' +/-H,-V states
d0			long	1 << 9				' increment destination by 1
d1			long	2 << 9				' increment destination by 2
d0s0			long	1 << 9 + 1			' increment destination and source by 1
address_mask		long	1023				' 1024 nibbles of colour RAM

' This code is pasted into scancode++ for the number of displayed columns
i_waitvid		waitvid	colorbuff, scanbuff
i_shr		if_c	shr	scanbuff, #8
i_waitvid0		waitvid	colorbuff, scanbuff
i_shr0		if_c	shr	scanbuff, #8
i_jmp_scanret		jmp	#scanret

			' Uninitialized data

screen_ptr		res	1
colour_ptr		res	1
font_ptr		res	1
screen_addr		res	1

gfxptr0			res	1
gfxptr1			res	1

cursor			res	1
cursor_first		res	1
cursor_last		res	1
scanlines		res	1
w			res	1
x			res	1
y			res	1
z			res	1

rowcount		res	1
colcount		res	1
horz_disp		res	1
horz_setup		res	1
xhf			res	1			' horizontal front porch
xhb			res	1			' horizontal back porch

scanbuff		res	44
colorbuff		res	44
scancode		res	2*44+1

			fit	$1f0

' 8 x 8 font - characters 0..255
'
' Each long holds four scan lines of a single character. The longs are arranged into
' groups of 256 which represent all characters (0..255). There are two groups which
' each contain a vertical half of all characters. They are ordered top to bottom.

font	long
  long $00000000,$0c0c0000,$30300000,$3c3c0000,$00000000,$0c0c0000,$30300000,$3c3c0000	' 1st half
  long $00000000,$0c0c0000,$30300000,$3c3c0000,$00000000,$0c0c0000,$30300000,$3c3c0000
  long $22140822,$22221c22,$22220022,$201c0022,$221c0022,$22220022,$1e22221c,$1008447e
  long $083e0808,$18061860,$18601806,$131010f0,$00324c00,$0408120c,$1212160a,$7e007e00
  long $00000000,$08080808,$00141414,$143e1414,$1c0a3c08,$08102606,$240a0a04,$00080808
  long $04040810,$10100804,$081c2a08,$3e080800,$00000000,$3e000000,$00000000,$08102000
  long $2a32221c,$08080c08,$1c20221c,$1810203e,$12141810,$201e023e,$1e020438,$1020203e
  long $1c22221c,$3c22221c,$08000000,$08000000,$02040810,$003e0000,$20100804,$1020221c
  long $3a2a221c,$22221408,$1e22221e,$0202221c,$2222221e,$0e02023e,$0e02023e,$3a02221c
  long $3e222222,$0808081c,$2020203e,$060a1222,$02020202,$2a2a3622,$322a2622,$2222221c
  long $1e22221e,$2222221c,$1e22221e,$1c02221c,$08082a3e,$22222222,$14222222,$2a222222
  long $08142222,$08142222,$0810223e,$0606061e,$08040200,$3030303c,$00221408,$00000000
  long $00100804,$201c0000,$221e0202,$221c0000,$223c2020,$221c0000,$040e0418,$223c0000
  long $221e0202,$080c0008,$20200020,$0a120202,$0808080c,$2a160000,$221e0000,$221c0000
  long $221e0000,$223c0000,$061a0000,$023c0000,$043e0404,$22220000,$22220000,$22220000
  long $14220000,$22220000,$103e0000,$02040418,$00080808,$2010100c,$0000324c,$ffffffff
  long $e0ffffff,$07ffffff,$e0e0e0e0,$07070707,$1f181818,$f8181818,$ff181818,$ff000000
  long $ff000000,$01010101,$80808080,$000000ff,$00000000,$818181ff,$18181818,$18000000
  long $3c3c3c3c,$ffff0000,$03031f1f,$c0f8f8f8,$78180606,$1e186060,$0f0f0303,$f0f0c0c0
  long $3c3c0000,$3c3c3030,$3f3f0000,$03030303,$c0c0c0c0,$3c181800,$3cc3c300,$00000000
  long $3c3cffff,$7e7e7e00,$0303ffff,$3c3c3c3c,$c3c3ffff,$c3c3c3c3,$ffff3c3c,$e7242424
  long $c3c3c3c3,$ffffffff,$ffff1818,$03030303,$c0c0ffff,$0f0fffff,$0f0ff0f0,$3f3f3f3f
  long $c3c3c3c3,$3f3f3f3f,$c3c3c3c3,$f0f0f0f0,$00000000,$0000ffff,$c0c0c0c0,$a5bd81ff
  long $666666e7,$3cc3c300,$18181818,$00000000,$00000000,$f0f0f0f0,$00000000,$00000f0f
  long $18ff6618,$ff00ff00,$ff181818,$cccc3333,$aa55aa55,$183c66c3,$7f7f7f36,$e7dbc381
  long $7f3e1c08,$2a1c1c08,$ffffffff,$cccccccc,$49494949,$f0e0c080,$0000ffff,$03030303
  long $ffff0000,$ffffffff,$00000000,$0000ffff,$f0f0f0f0,$0f0f0f0f,$0f0f0f0f,$00ff00ff
  long $6d6d6d6d,$ff000000,$00000000,$c0c0c0c0,$44881020,$33333333,$22110804,$3060c080
  long $0f070301,$18181818,$1f3f7fff,$180c0603,$f8fcfeff,$3c7effff,$ffff7e3c,$99999999
  long $995a3c18,$998181ff,$18000303,$c3c37e3c,$00000000,$00000000,$f0f0e0c0,$f0101010
  long $0f080808,$3c2424e7,$7f3e1c08,$18a54224,$00000000,$0f0f0703,$18000000,$00666600
  long $8181c3ff,$1800c3c3,$66ffff66,$18181818,$18180000,$ff020408,$66006666,$ff402010

  long $00000000,$00000000,$00000000,$00000000,$00000c0c,$00000c0c,$00000c0c,$00000c0c	' 2nd half
  long $00003030,$00003030,$00003030,$00003030,$00003c3c,$00003c3c,$00003c3c,$00003c3c
  long $0022223e,$001c2222,$001c2222,$003c223c,$001c2222,$002c3222,$021e2222,$007e4408
  long $003e0008,$007e0060,$007e0006,$10181412,$0000324c,$0000001e,$00000012,$00007e00
  long $00000000,$00080008,$00000000,$0014143e,$00081e28,$00303204,$002c121a,$00000000
  long $00100804,$00040810,$00082a1c,$00000808,$04080800,$00000000,$00080000,$00000204
  long $001c2226,$00080808,$003e0202,$001c2220,$0010103e,$001c2220,$001c2222,$00020408
  long $001c2222,$000e1020,$00000800,$04080800,$00100804,$0000003e,$00040810,$00080008
  long $003c021a,$0022223e,$001e2222,$001c2202,$001e2222,$003e0202,$00020202,$001c2222
  long $00222222,$001c0808,$001c2220,$0022120a,$003e0202,$00222222,$00222222,$001c2222
  long $00020202,$002c122a,$0022120a,$001c2220,$00080808,$001c2222,$00080814,$00142a2a
  long $00222214,$00080808,$003e2204,$001e0606,$00002010,$003c3030,$00000000,$3e000000
  long $00000000,$003c223c,$001e2222,$001c2202,$003c2222,$001c023e,$04040404,$1c203c22
  long $00222222,$001c0808,$18242020,$00120a06,$001c0808,$002a2a2a,$00222222,$001c2222
  long $02021e22,$20203c22,$00020202,$001e201c,$00380404,$002c3222,$00081422,$00142a2a
  long $00221408,$1c203c22,$003e0408,$00180404,$00080808,$000c1010,$00000000,$ffffffff
  long $e0e0e0e0,$07070707,$ffffffe0,$ffffff07,$1818181f,$181818f8,$000000ff,$181818ff
  long $000000ff,$01010101,$80808080,$00000000,$ff000000,$ff818181,$18181818,$00000018
  long $3c3c3c3c,$0000ffff,$03030303,$c0c0c0c0,$06061878,$6060181e,$03030f0f,$c0c0f0f0
  long $00003c3c,$0c0c3c3c,$0000fcfc,$3f3f0303,$fcfcfcc0,$00c3c33c,$0018183c,$ffff3c3c
  long $00000000,$007e7e7e,$ffff0303,$3c3c3c3c,$c3c3c3c3,$ffffc3c3,$3c3c3c3c,$242424e7
  long $c3c3c3c3,$1818ffff,$ffffffff,$03030303,$ffffc0c0,$f0f00f0f,$ffff0f0f,$c3c3c3c3
  long $3f3f3f3f,$3f3f3f3f,$fcfcfcfc,$cccccccc,$ffff0000,$00000000,$c0c0c0c0,$ff81bda5
  long $ffffffff,$00c3c33c,$18181818,$3f3f0000,$fcfc0000,$00000000,$0f0f0f0f,$00000000
  long $81814224,$ff00ff00,$181818ff,$cccc3333,$aa55aa55,$81c3663c,$081c3e3e,$0081c3db
  long $3e08367f,$3e082a7f,$ffffffff,$cccccccc,$49494949,$fffefcf8,$0000ffff,$03030303
  long $ffff0000,$00000000,$ffffffff,$00000000,$f0f0f0f0,$0f0f0f0f,$f0f0f0f0,$00ff00ff
  long $6d6d6d6d,$000000ff,$ffff0000,$c0c0c0c0,$04081122,$33333333,$20108844,$03060c18
  long $ff7f3f1f,$18181818,$0103070f,$80c06030,$80c0e0f0,$00000000,$3c7effff,$183c5a99
  long $18181818,$ff818199,$c0c00018,$3c7ec3c3,$101010f0,$0808080f,$c0e0f0f0,$00000000
  long $00000000,$8181c366,$081c3e7f,$2442a518,$ffff7e3c,$03070f0f,$00000018,$00666600
  long $ffc38181,$c3c30018,$66ffff66,$183c5a99,$00181800,$080402ff,$66660066,$102040ff

'' MIT LICENSE
{{
'	Permission is hereby granted, free of charge, to any person obtaining
'	a copy of this software and associated documentation files
'	(the "Software"), to deal in the Software without restriction,
'	including without limitation the rights to use, copy, modify, merge,
'	publish, distribute, sublicense, and/or sell copies of the Software,
'	and to permit persons to whom the Software is furnished to do so,
'	subject to the following conditions:
'
'	The above copyright notice and this permission notice shall be included
'	in all copies or substantial portions of the Software.
'
'	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
'	EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
'	MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
'	IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
'	CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
'	TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
'	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
}}
