''***********************************************
''*	VGA High-Res Text Driver v1.0		*
''*	Author: Chip Gracey			*
''*	Copyright (c) 2006 Parallax, Inc.	*
''*	See end of file for terms of use.	*
''***********************************************
''
'' This object generates a 768x512 VGA signal which contains 64 columns x 16
'' rows of 6x16 characters. Each row can have a unique forground/background
'' color combination and each character can be inversed. There are also two
'' cursors which can be independently controlled (ie. mouse and keyboard). A
'' sync indicator signals each time the screen is refreshed (you may ignore).
''
'' You must provide buffers for the screen, colors, cursors, and sync. Once
'' started, all interfacing is done via memory. To this object, all buffers are
'' read-only, with the exception of the sync indicator which gets written with
'' -1. You may freely write all buffers to affect screen appearance. Have fun!
''

CON

' 800 x 600 @ 75Hz settings: 64 x 16 characters

	hp	= 768		'horizontal pixels
	vp	= 512		'vertical pixels
	hf	= 96		'horizontal front porch pixels
	hs	= 128		'horizontal sync pixels
	hb	= 140'96	'horizontal back porch pixels
	vf	= 55		'vertical front porch lines
	vs	= 4		'vertical sync lines
	vb	= 57		'vertical back porch lines
	hn	= 1		'horizontal normal sync state (0|1)
	vn	= 1		'vertical normal sync state (0|1)
	pr	= 50		'pixel rate in MHz at 80MHz system clock (5MHz granularity)

' columns and rows

	cols = 64
	rows = 16


VAR long cog[2]

PUB start(BasePin, ScreenPtr, SyncPtr) : okay | i, j

'' Start VGA driver - starts two COGs
'' returns false if two COGs not available
''
''	BasePin = VGA starting pin (0, 8, 16, 24, etc.)
''
''	ScreenPtr = Pointer to 8,192 bytes containing ASCII codes for each of the
''		128x64 screen characters. Each byte's top bit controls color
''		inversion while the lower seven bits provide the ASCII code.
''		Screen memory is arranged left-to-right, top-to-bottom.
''
''		screen byte example: %1_1000001 = inverse "A"
''
''	ColorPtr = Pointer to 64 words which define the foreground and background
''		colors for each row. The lower byte of each word contains the
''		foreground RGB data for that row, while the upper byte
''		contains the background RGB data. The RGB data in each byte is
''		arranged as %RRGGBB00 (4 levels each).
''
''		color word example: %%0020_3300 = gold on blue
''
''	CursorPtr = Pointer to 6 bytes which control the cursors:
''
''		bytes 0,1,2: X, Y, and MODE of cursor 0
''		bytes 3,4,5: X, Y, and MODE of cursor 1
''
''		X and Y are in terms of screen characters
''		(left-to-right, top-to-bottom)
''
''		MODE uses three bottom bits:
''
''			%x00 = cursor off
''			%x01 = cursor on
''			%x10 = cursor on, blink slow
''			%x11 = cursor on, blink fast
''			%0xx = cursor is solid block
''			%1xx = cursor is underscore
''
''		cursor example: 127, 63, %010 = blinking block in lower-right
''
''	SyncPtr = Pointer to long which gets written with -1 upon each screen
''		refresh. May be used to time writes/scrolls, so that chopiness
''		can be avoided. You must clear it each time if you want to see
''		it re-trigger.

	'if driver is already running, stop it
	stop

	'implant pin settings
	reg_vcfg := $200000FF + (BasePin & %111000) << 6
	i := $FF << (BasePin & %011000)
	j := BasePin & %100000 == 0
	reg_dira := i & j
	reg_dirb := i & !j

	'implant CNT value to sync COGs to
	sync_cnt := cnt + $20000

	'implant pointers
	screen_base := ScreenPtr
	font_base := @font

	'implant unique settings and launch first COG
	vf_lines.byte := vf
	vb_lines.byte := vb
	font_quarter := 1
	cog[1] := cognew(@d0, SyncPtr) + 1

	'allow time for first COG to launch
	waitcnt($4000 + cnt)

	'differentiate settings and launch second COG
	vf_lines.byte := vf+8
	vb_lines.byte := vb-8
	font_quarter := 0
	cog[0] := cognew(@d0, SyncPtr) + 1

	'if both COGs launched, return true
	if cog[0] and cog[1]
		return 1

	'else, stop any launched COG and return false
	else
		stop


PUB stop | i

'' Stop VGA driver - frees two COGs

	repeat i from 0 to 1
		if cog[i]
			cogstop(cog[i]~ - 1)


PUB write_chargen(offs, src, size) | addr, i
'' Write the character generator
	repeat i from offs to offs + size - 1
		addr := (i & $003) + ((i & $ff0) >> 2) + ((i & $00c) * 256)
		byte[@font][addr] := byte[src++] >< 8

CON

	#1, scanbuff[128], scancode[128*2-1+3], maincode	'enumerate COG RAM usage

	main_size = $1F0 - maincode				'size of main program

	hv_inactive = (hn << 1 + vn) * $0101			'H,V inactive states


DAT

'*****************************************************
'* Assembly language VGA high-resolution text driver *
'*****************************************************

' This program runs concurrently in two different COGs.
'
' Each COG's program has different values implanted for front-porch lines and
' back-porch lines which surround the vertical sync pulse lines. This allows
' timed interleaving of their active display signals during the visible portion
' of the field scan. Also, they are differentiated so that one COG displays
' even four-line groups while the other COG displays odd four-line groups.
'
' These COGs are launched in the PUB 'start' and are programmed to synchronize
' their PLL-driven video circuits so that they can alternately prepare sets of
' four scan lines and then display them. The COG-to-COG switchover is seemless
' due to two things: exact synchronization of the two video circuits and the
' fact that all COGs' driven output states get OR'd together, allowing one COG
' to output lows during its preparatory state while the other COG effectively
' drives the pins to create the visible and sync portions of its scan lines.
' During non-visible scan lines, both COGs output together in unison.
'
' COG RAM usage:	$000	= d0 - used to inc destination fields for indirection
'		 $001-$080 = scanbuff - longs which hold 4 scan lines
'		 $081-$182 = scancode - stacked WAITVID/SHR for fast display
'		 $183-$1EF = maincode - main program loop which drives display

			org	0				' set origin to $000 for start of program

d0			long	1 << 9				' d0 always resides here at $000, executes as NOP


' Initialization code and data - after execution, space gets reused as scanbuff

			' Move main program into maincode area

:move			mov	$1EF,main_begin+main_size-1
			sub	:move,d0s0			' (do reverse move to avoid overwrite)
			djnz	main_ctr,#:move

			' Build scanbuff display routine into scancode
			' org	scancode
:waitvid1		mov	scancode+0,i0			' waitvid color,scanbuff+0
			add	:waitvid1,d0			' waitvid color,scanbuff+1
			add	i0,#1				' ...
			djnz	scan_ctr,#:waitvid1		' waitvid color,scanbuff+cols-1

			mov	scancode+1*cols+0,i3		' mov	vscl,#hf
			mov	scancode+1*cols+1,i4		' waitvid hvsync,#0
			mov	scancode+1*cols+2,i5		' mov	vscl,#hs
			mov	scancode+1*cols+3,i6		' waitvid hvsync,#1
			mov	scancode+1*cols+4,i7		' mov	vscl,#hb
			mov	scancode+1*cols+5,i8		' waitvid hvsync,#0
			mov	scancode+1*cols+6,i9		' mov	vscl,vscl_chr

			mov	scan_ctr,#cols
:waitvid2		mov	scancode+1*cols+7+0,i1		' waitvid color,scanbuff+0
:ror			mov	scancode+1*cols+7+1,i2		' ror	scanbuff+0,#8
			add	:waitvid2,d1			' waitvid color,scanbuff+1
			add	:ror,d1				' ror	scanbuff+1,#8
			add	i1,#1				' ...
			add	i2,d0				' waitvid color,scanbuff+cols-1
			djnz	scan_ctr,#:waitvid2		' overwrite last ror with the following

			mov	scancode+3*cols+7-1,i3		' mov	vscl,#hf
			mov	scancode+3*cols+7+0,i4		' waitvid hvsync,#0
			mov	scancode+3*cols+7+1,ix		' jmp	#scanret

			'Init I/O registers and sync COGs' video circuits

			mov	dira,reg_dira			' set pin directions
			mov	dirb,reg_dirb
			movi	frqa,#(pr / 5) << 2		' set pixel rate
			mov	vcfg,reg_vcfg			' set video configuration
			mov	vscl,#1				' set video to reload on every pixel
			waitcnt sync_cnt,colormask		' wait for start value in cnt, add ~1ms
			movi	ctra,#%00001_110		' COGs in sync! enable PLLs now - NCOs locked!
			waitcnt sync_cnt,#0			' wait ~1ms for PLLs to stabilize - PLLs locked!
			mov	vscl,#100			' insure initial WAITVIDs lock cleanly

			'Jump to main loop

			jmp	#vsync				' jump to vsync - WAITVIDs will now be locked!

			' Data

d0s0			long	1 << 9 + 1
d1			long	2 << 9
main_ctr		long	main_size
scan_ctr		long	cols

i0			waitvid x,scanbuff+0
i1			waitvid x,scanbuff+0
i2			ror	scanbuff+0,#8
i3			mov	vscl,#hf
i4			waitvid hvsync,#0
i5			mov	vscl,#hs			' do horizontal sync pixels
i6			waitvid hvsync,#1			' #1 makes hsync active
i7			mov	vscl,#hb			' do horizontal back porch pixels
i8			waitvid hvsync,#0			' #0 makes hsync inactive
i9			mov	vscl,vscl_chr			' do character pixels
ix			jmp	#scanret

reg_dira		long	0				' set at runtime
reg_dirb		long	0				' set at runtime
reg_vcfg		long	0				' set at runtime
sync_cnt		long	0				' set at runtime

			'Directives

			fit	scancode			' make sure initialization code and data fit
main_begin		org	maincode			' main code follows (gets moved into maincode)


' Main loop, display field - each COG alternately builds and displays four scan lines

vsync			mov	x,#vs				' do vertical sync lines
			call	#blank_vsync

			mov	screen_ptr,screen_base		' reset screen pointer to upper-left character

vb_lines		mov	x,#vb				' do vertical back porch lines (# set at runtime)
			call	#blank_vsync

			mov	row, #0				' reset row counter for cursor insertion
			mov	fours, #rows * 4 / 2		' set number of 4-line builds for whole screen

			' Build four scan lines into scanbuff

fourline		mov	font_ptr, font_quarter		' get address of appropriate font section
			shl	font_ptr, #8+2			' shl 8 for 256 characters, 2 for indexing longs
			add	font_ptr, font_base

			movd	:pixa, #scanbuff-1		' reset scanbuff address (pre-decremented)

			mov	y, #4				' must build scanbuff in four sections because
			mov	vscl, vscl_line2x		' ..pixel counter is limited to twelve bits

:quarterrow		waitvid underscore, #0			' output lows to let other COG drive VGA pins
			mov	x, #cols/4			' ..for 2 scan lines, ready for a quarter row

:column			rdbyte	z, screen_ptr			' get character from screen memory
			cmp	z, #$20			wc	' TRS80 has a missing bit on the video RAM
		if_c	add	z, #$40				' so we adjust character codes below $20
			shl	z, #2				' get character into bits 9..2
			add	z, font_ptr			' add font section address to point to 8*4 pixels
			add	:pixa, d0			' increment scanbuff destination addresses
			add	screen_ptr, #1			' increment screen memory address
:pixa			rdlong	scanbuff, z			' read pixel long (8*4) into scanbuff
			djnz	x, #:column			' another character in this half-row?

			djnz	y, #:quarterrow			' loop to do next quarter row, time for WAITVID

			sub	screen_ptr, #cols		' back up to start of same row in screen memory

			' Display four scan lines from scanbuff

			mov	x, color			' get color pattern
			or	x, hv				' insert inactive hsync and vsync states

			mov	y, #4				' ready for four times two scan lines

scanline		mov	vscl, vscl_chr			' set pixel rate for characters
			jmp	#scancode			' jump to scanbuff display routine in scancode
scanret			mov	vscl, #hs			' do horizontal sync pixels
			waitvid hvsync, #1			' #1 makes hsync active
			mov	vscl, #hb			' do horizontal back porch pixels
			waitvid hvsync, #0			' #0 makes hsync inactive
			ror	scanbuff+cols-1, #8		' rotate last column's pixels right by 8
			djnz	y, #scanline			' another scan line?

			' Next group of four scan lines

			add	font_quarter, #2	 	' if font_quarter + 2 => 4, subtract 4 (new row)
			cmpsub	font_quarter, #4	wc	' c=0 for same row, c=1 for new row
	if_c		add	screen_ptr, #cols		' if new row, advance screen pointer
	if_c		add	row, #1				' if new row, increment row counter
			djnz	fours, #fourline		' another 4-line build/display?

			'Visible section done, do vertical sync front porch lines

			wrlong	longmask, par			' write -1 to refresh indicator

vf_lines		mov	x, #vf				' do vertical front porch lines (# set at runtime)
			call	#blank

			jmp	#vsync				' new field, loop to vsync

			' Subroutine - do blank lines

blank_vsync		xor	hvsync, #$101			' flip vertical sync bits

blank			mov	vscl, hx			' do blank pixels
			waitvid hvsync,#0
			mov	vscl, #hf			' do horizontal front porch pixels
			waitvid hvsync,#0
			mov	vscl, #hs			' do horizontal sync pixels
			waitvid hvsync,#1
			mov	vscl, #hb			' do horizontal back porch pixels
			waitvid hvsync,#0
			djnz	x, #blank			' another line?
blank_ret
blank_vsync_ret	 ret

			' Data

screen_base		long	0				' set at runtime (3 contiguous longs)
font_base		long	0				' set at runtime
font_quarter		long	0				' set at runtime

hx			long	hp				' visible pixels per scan line
vscl_line2x		long	(hp + hf + hs + hb) * 2 	' total number of pixels per 2 scan lines
vscl_chr		long	2 << 12 + 12			' 2 clocks per pixel and 12 pixels per set
colormask		long	$FCFC				' mask to isolate R,G,B bits from H,V
longmask		long	$FFFFFFFF			' all bits set
slowbit			long	1 << 25				' cnt mask for slow cursor blink
fastbit			long	1 << 24				' cnt mask for fast cursor blink
underscore		long	$FFFF0000			' underscore cursor pattern
hv			long	hv_inactive			' -H,-V states
hvsync			long	hv_inactive ^ $200		' +/-H,-V states
color			long	%00100000_00000000

			' Uninitialized data

screen_ptr		res	1
font_ptr		res	1

x			res	1
y			res	1
z			res	1

row			res	1
fours			res	1


' 8 x 16 font - characters 0..255
'
' Each long holds four scan lines of a single character. The longs are arranged into
' groups of 256 which represent all characters (0..255). There are four groups which
' each contain a vertical quarter of all characters. They are ordered top to bottom.

font	long
	long $22223e00,$02023e00,$08080800,$20202000,$10080400,$36223e00,$10200000,$22221c00
	long $1e040800,$10080000,$00003e00,$08080000,$1c2a0800,$04080000,$36221c00,$22221c00
	long $22223e00,$2a2a1c00,$22221c00,$22221c00,$2a2a1c00,$10280000,$14141c00,$20202000
	long $14223e00,$1c080800,$02221c00,$22221c00,$2a2a3e00,$22223e00,$22223e00,$2a2a3e00
	long $00000000,$08080800,$14141400,$3e141400,$0a3c0800,$10260600,$0a0a0400,$040c0c00
	long $04081000,$10080400,$1c2a0800,$08080000,$00000000,$00000000,$00000000,$10200000
	long $32221c00,$080c0800,$20221c00,$20221c00,$14181000,$1e023e00,$02041800,$10203e00
	long $22221c00,$22221c00,$0c0c0000,$0c000000,$04081000,$3e000000,$10080400,$20221c00
	long $20221c00,$22140800,$24241e00,$02221c00,$24241e00,$02023e00,$02023e00,$02023c00
	long $22222200,$08081c00,$20202000,$0a122200,$02020200,$2a362200,$2a262200,$22221c00
	long $22221e00,$22221c00,$22221e00,$02221c00,$08083e00,$22222200,$22222200,$22222200
	long $14222200,$14222200,$10203e00,$2a1c0800,$08080800,$04080000,$10080000,$00000000
	long $08181800,$1c000000,$1a020200,$1c000000,$2c202000,$1c000000,$08281000,$2c000000
	long $1a020200,$0c000800,$20000000,$12020200,$08080c00,$16000000,$1a000000,$1c000000
	long $1a000000,$2c000000,$1a000000,$3c000000,$3e080800,$22000000,$22000000,$22000000
	long $22000000,$22000000,$3e000000,$08081000,$08080800,$08080400,$102a0400,$142a1400
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f
	long $00000000,$07070707,$38383838,$3f3f3f3f,$00000000,$07070707,$38383838,$3f3f3f3f

	long $3e222222,$02020202,$3e080808,$3e202020,$2010083c,$3e22362a,$0002060a,$3614143e
	long $20202824,$0008103e,$3e00003e,$00081c2a,$081c2a08,$0008043e,$1c22362a,$1c22222a
	long $3e22223e,$1c22223a,$1c2a2a3a,$1c2a2a2e,$1c22222e,$0002062a,$36141414,$2020203e
	long $3e221408,$0808081c,$08000804,$1c22223e,$3e22222e,$3e2a2a2e,$3e2a2a3a,$3e22223a
	long $00000000,$08000808,$00000000,$14143e14,$081e281c,$30320408,$2c122a04,$00000002
	long $10080404,$04081010,$082a1c3e,$0008083e,$0c0c0000,$0000003e,$0c0c0000,$00020408
	long $1c22262a,$1c080808,$3e02021c,$1c222018,$10103e12,$1c222020,$1c22221e,$02020408
	long $1c22221c,$0c10203c,$000c0c00,$0c0c000c,$10080402,$00003e00,$04081020,$08000810
	long $1c2a2a2c,$22223e22,$1e24241c,$1c220202,$1e242424,$3e02020e,$0202020e,$3c222232
	long $2222223e,$1c080808,$1c222020,$22120a06,$3e020202,$2222222a,$22222232,$1c222222
	long $0202021e,$2c122a22,$22120a1e,$1c22201c,$08080808,$1c222222,$08081414,$22362a22
	long $22221408,$08080808,$3e020408,$08080808,$081c2a08,$0008043e,$0008103e,$3e000000
	long $00000010,$3c223c20,$1a262226,$1c220222,$2c322232,$1c023e22,$0808081c,$202c3232
	long $22222226,$1c080808,$20202000,$120a060a,$1c080808,$2a2a2a2a,$22222226,$1c222222
	long $1a262226,$2c322232,$02020226,$1e201c02,$10280808,$2c322222,$08142222,$142a2a22
	long $22140814,$203c2222,$3e040810,$10080804,$08080800,$04080810,$00000000,$142a142a
	long $00000000,$00000007,$00000038,$0000003f,$07070700,$07070707,$07070738,$0707073f
	long $38383800,$38383807,$38383838,$3838383f,$3f3f3f00,$3f3f3f07,$3f3f3f38,$3f3f3f3f
	long $00000000,$00000007,$00000038,$0000003f,$07070700,$07070707,$07070738,$0707073f
	long $38383800,$38383807,$38383838,$3838383f,$3f3f3f00,$3f3f3f07,$3f3f3f38,$3f3f3f3f
	long $00000000,$00000007,$00000038,$0000003f,$07070700,$07070707,$07070738,$0707073f
	long $38383800,$38383807,$38383838,$3838383f,$3f3f3f00,$3f3f3f07,$3f3f3f38,$3f3f3f3f
	long $00000000,$00000007,$00000038,$0000003f,$07070700,$07070707,$07070738,$0707073f
	long $38383800,$38383807,$38383838,$3838383f,$3f3f3f00,$3f3f3f07,$3f3f3f38,$3f3f3f3f
	long $00000000,$00000007,$00000038,$0000003f,$07070700,$07070707,$07070738,$0707073f
	long $38383800,$38383807,$38383838,$3838383f,$3f3f3f00,$3f3f3f07,$3f3f3f38,$3f3f3f3f
	long $00000000,$00000007,$00000038,$0000003f,$07070700,$07070707,$07070738,$0707073f
	long $38383800,$38383807,$38383838,$3838383f,$3f3f3f00,$3f3f3f07,$3f3f3f38,$3f3f3f3f
	long $00000000,$00000007,$00000038,$0000003f,$07070700,$07070707,$07070738,$0707073f
	long $38383800,$38383807,$38383838,$3838383f,$3f3f3f00,$3f3f3f07,$3f3f3f38,$3f3f3f3f
	long $00000000,$00000007,$00000038,$0000003f,$07070700,$07070707,$07070738,$0707073f
	long $38383800,$38383807,$38383838,$3838383f,$3f3f3f00,$3f3f3f07,$3f3f3f38,$3f3f3f3f

	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000204,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000204,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00001c22
	long $00000000,$00000000,$00001c22,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000202,$00002020,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00001c22,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000707,$00000707,$00000707,$00000707
	long $00003838,$00003838,$00003838,$00003838,$00003f3f,$00003f3f,$00003f3f,$00003f3f
	long $07070000,$07070000,$07070000,$07070000,$07070707,$07070707,$07070707,$07070707
	long $07073838,$07073838,$07073838,$07073838,$07073f3f,$07073f3f,$07073f3f,$07073f3f
	long $38380000,$38380000,$38380000,$38380000,$38380707,$38380707,$38380707,$38380707
	long $38383838,$38383838,$38383838,$38383838,$38383f3f,$38383f3f,$38383f3f,$38383f3f
	long $3f3f0000,$3f3f0000,$3f3f0000,$3f3f0000,$3f3f0707,$3f3f0707,$3f3f0707,$3f3f0707
	long $3f3f3838,$3f3f3838,$3f3f3838,$3f3f3838,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f
	long $00000000,$00000000,$00000000,$00000000,$00000707,$00000707,$00000707,$00000707
	long $00003838,$00003838,$00003838,$00003838,$00003f3f,$00003f3f,$00003f3f,$00003f3f
	long $07070000,$07070000,$07070000,$07070000,$07070707,$07070707,$07070707,$07070707
	long $07073838,$07073838,$07073838,$07073838,$07073f3f,$07073f3f,$07073f3f,$07073f3f
	long $38380000,$38380000,$38380000,$38380000,$38380707,$38380707,$38380707,$38380707
	long $38383838,$38383838,$38383838,$38383838,$38383f3f,$38383f3f,$38383f3f,$38383f3f
	long $3f3f0000,$3f3f0000,$3f3f0000,$3f3f0000,$3f3f0707,$3f3f0707,$3f3f0707,$3f3f0707
	long $3f3f3838,$3f3f3838,$3f3f3838,$3f3f3838,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f

	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $07070707,$07070707,$07070707,$07070707,$07070707,$07070707,$07070707,$07070707
	long $07070707,$07070707,$07070707,$07070707,$07070707,$07070707,$07070707,$07070707
	long $38383838,$38383838,$38383838,$38383838,$38383838,$38383838,$38383838,$38383838
	long $38383838,$38383838,$38383838,$38383838,$38383838,$38383838,$38383838,$38383838
	long $3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f
	long $3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000,$00000000
	long $07070707,$07070707,$07070707,$07070707,$07070707,$07070707,$07070707,$07070707
	long $07070707,$07070707,$07070707,$07070707,$07070707,$07070707,$07070707,$07070707
	long $38383838,$38383838,$38383838,$38383838,$38383838,$38383838,$38383838,$38383838
	long $38383838,$38383838,$38383838,$38383838,$38383838,$38383838,$38383838,$38383838
	long $3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f
	long $3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f,$3f3f3f3f

'' MIT LICENSE
{{
'	Permission is hereby granted, free of charge, to any person obtaining
'	a copy of this software and associated documentation files
'	(the "Software"), to deal in the Software without restriction,
'	including without limitation the rights to use, copy, modify, merge,
'	publish, distribute, sublicense, and/or sell copies of the Software,
'	and to permit persons to whom the Software is furnished to do so,
'	subject to the following conditions:
'
'	The above copyright notice and this permission notice shall be included
'	in all copies or substantial portions of the Software.
'
'	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
'	EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
'	MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
'	IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
'	CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
'	TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
'	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
}}
