''***************************************
''*	PS/2 Keyboard Driver v1.0.1	*
''*	Author: Chip Gracey		*
''*	Copyright (c) 2004 Parallax, Inc.	*
''*	See end of file for terms of use.	*
''***************************************

{-----------------REVISION HISTORY-----------------
 v1.0.1 - Updated 6/15/2006 to work with Propeller Tool 0.96}

CON
	stat_extended	= 1		' extended keycode flag
	stat_break	= 2		' break code flag
	stat_config	= 4		' configure keyboard flag
	stat_reset	= 8		' reset keyboard flag

VAR
	long	cog
	long	par_states[8]	'key states (256 bits)	read-only
	long	par_present	'keyboard present	read-only
  long  par_tail        'key buffer tail        read/write      (19 contiguous longs)
  long  par_head        'key buffer head        read-only
  long  par_keys[8]     'key buffer (16 words)  read-only       (also used to pass initial parameters)

PUB start(dpin, cpin) : okay

'' Start keyboard driver - starts a cog
'' returns false if no cog available
''
''	dpin	= data signal on PS/2 jack
''	cpin	= clock signal on PS/2 jack
''
''	use 100-ohm resistors between pins and jack
''	use 10K-ohm resistors to pull jack-side signals to VDD
''	connect jack-power to 5V, jack-gnd to VSS
''
'' all lock-keys will be enabled, NumLock will be initially 'on',
'' and auto-repeat will be set to 15cps with a delay of .5s

	okay := startx(dpin, cpin, %0_000_100, %01_01000)

PUB startx(dpin, cpin, locks, auto) : okay

'' Like start, but allows you to specify lock settings and auto-repeat
''
''	locks = lock setup
''		bit 6 disallows shift-alphas (case set soley by CapsLock)
''		bits 5..3 disallow toggle of NumLock/CapsLock/ScrollLock state
''		bits 2..0 specify initial state of NumLock/CapsLock/ScrollLock
''		(eg. %0_001_100 = disallow ScrollLock, NumLock initially 'on')
''
''	auto	= auto-repeat setup
''		bits 6..5 specify delay (0=.25s, 1=.5s, 2=.75s, 3=1s)
''		bits 4..0 specify repeat rate (0=30cps..31=2cps)
''		(eg %01_00000 = .5s delay, 30cps repeat)

	stop
	longmove(@par_states, @dpin, 4)
	okay := cog := cognew(@entry, @par_states) + 1


PUB stop
'' Stop keyboard driver - frees a cog
	if cog
		cogstop(cog~ -	1)
	longfill(@par_states, 0, 8+1+1)


PUB present : truefalse
'' Check if keyboard present - valid ~2s after start
'' returns t|f
	truefalse := -par_present

PUB keystate(k) : state
'' Get the state of a particular key
'' returns t|f
	state := -(par_states[k >> 5] >> k & 1)

PUB states_ptr
	return @par_states	' returns the location of par_states longs

DAT

'******************************************
'* Assembly language PS/2 keyboard driver *
'******************************************

			org	0
'
'
' Entry
'
entry			movd	:par, #_dpin			' load input parameters _dpin/_cpin/_locks/_auto
			mov	x, par
			mov	y, #4
			mov	t, #0
:par			rdlong	0, x
			add	:par, dlsb
			wrlong	t, x				' clear the states
			add	x, #4
			djnz	y, #:par

			mov	dmask, #1			' set pin masks
			shl	dmask, _dpin			' 0..31 (ignores bit #5)
			mov	cmask, #1
			shl	cmask, _cpin			' 0..31 (ignores bit #5)

			test	_dpin, #$20	wc		' modify port registers within code
			muxc	_d1, dlsb
			muxc	_d2, dlsb
			muxc	_d3, #1
			muxc	_d4, #1
			test	_cpin, #$20	wc
			muxc	_c1, dlsb
			muxc	_c2, dlsb
			muxc	_c3, #1
'
'
' Reset keyboard
'
reset			mov	dira, #0			' reset directions
			mov	dirb, #0

			movd	:par, #_states			' reset output parameters _states[8]/_present
			mov	x, #8+1
:par			mov	0, #0
			add	:par, dlsb
			djnz	x, #:par

			mov	stat, #stat_reset		' set reset flag
'
'
' Update parameters
'
update			movd	:par, #_states			' update output _states[8]/_present
			mov	x, par
			mov	y, #8+1
:par			wrlong	0, x
			add	:par, dlsb
			add	x, #4
			djnz	y, #:par

			mov	x, _states+1			' get states $20-$3f
			shr	x, #24				' get states $38-$3f
			cmp	x, #%01100010		wz	' Ctrl + Alt + Delete?
		if_nz	jmp	#:not_reset
			mov	x, #$80
			clkset	x				' reset the propeller

:not_reset		test	stat, #stat_reset	wc	' if reset flag, transmit reset command
		if_c	mov	data, #$ff
		if_c	call	#transmit
'
'
' Get scancode
'
newcode
			mov	stat, #0			' reset state

:same
			call	#receive			' receive byte from keyboard

			cmp	data, #$83+1		wc	' scancode?

	if_nc		cmp	data, #$AA		wz	' powerup/reset?
	if_nc_and_z	jmp	#configure

	if_nc		cmp	data, #$E0		wz	' extended?
	if_nc_and_z	or	stat, #stat_extended
	if_nc_and_z	jmp	#:same

	if_nc		cmp	data, #$F0		wz	' released?
	if_nc_and_z	or	stat, #stat_break
	if_nc_and_z	jmp	#:same

	if_nc		jmp	#newcode			' unknown, ignore
'
'
' Translate scancode and enter into buffer
'
			test	stat, #stat_extended	wc	' lookup code with extended flag
			rcl	data, #1
			call	#look

			cmp	data, #$ff		wz	' if unknown, ignore
		if_z	jmp	#newcode

			mov	t, _states+6			' remember lock keys in _states

			mov	x, data				' set/clear key bit in _states
			shr	x, #5
			add	x, #_states
			movd	:reg, x
			mov	y, #1
			shl	y, data
			test	stat,#stat_break	wc
:reg			muxnc	0, y

			test	stat, #stat_config	wc	' if not configure flag, done
		if_nc	jmp	#update				' else configure to update leds
'
'
' Configure keyboard
'
configure
			mov	data, #$F3			' set keyboard auto-repeat
			call	#transmit
			mov	data, _auto
			and	data, #%11_11111
			call	#transmit

			mov	data, #$ED			' set keyboard lock-leds
			call	#transmit
			mov	data, _locks
			rev	data, #-3 & $1F
			test	data, #%100		wc
			rcl	data, #1
			and	data, #%111
			call	#transmit

			mov	x, _locks			' insert locks into _states
			and	x, #%111
			shl	_states+7, #3
			or	_states+7, x
			ror	_states+7, #3

			mov	_present, #1			' set _present

			jmp	#update				' done
'
'
' Lookup byte in table
'
look
			ror	data, #2			' perform lookup
			movs	:reg, data
			add	:reg, #table
			shr	data, #27
			mov	x, data
:reg			mov	data, 0-0
			shr	data, x
			jmp	#rand				' isolate byte
'
'
' Transmit byte to keyboard
'
transmit
_c1			or	dira,cmask			' pull clock low
			movs	napshr,#13			' hold clock for ~128us (must be >100us)
			call	#nap
_d1			or	dira,dmask			' pull data low
			movs	napshr,#18			' hold data for ~4us
			call	#nap
_c2			xor	dira,cmask			' release clock

			test	data, #$0FF		wc	' append parity and stop bits to byte
			muxnc	data, #$100
			or	data, dlsb

			mov	x, #10				' ready 10 bits
transmit_bit
			call	#wait_c0			' wait until clock low
			shr	data, #1		wc	' output data bit
_d2			muxnc	dira,dmask
			mov	wcond, c1			' wait until clock high
			call	#wait
			djnz	x, #transmit_bit		' another bit?

			mov	wcond, c0d0			' wait until clock and data low
			call	#wait
			mov	wcond, c1d1			' wait until clock and data high
			call	#wait

			call	#receive_ack			' receive ack byte with timed wait
			cmp	data, #$FA		wz	' if ack error, reset keyboard
		if_nz	jmp	#reset
transmit_ret
			ret
'
'
' Receive byte from keyboard
'
receive			test	_cpin, #$20		wc	' wait indefinitely for initial clock low
			waitpne cmask,cmask
receive_ack		mov	x,#11				' ready 11 bits
receive_bit		call	#wait_c0			' wait until clock low
			movs	napshr,#16			' pause ~16us
			call	#nap
_d3			test	dmask, ina		wc	' input data bit
			rcr	data, #1
			mov	wcond, c1			' wait until clock high
			call	#wait
			djnz	x, #receive_bit			' another bit?
			shr	data, #22			' align byte
			test	data, #$1FF		wc	' if parity error, reset keyboard
		if_nc	jmp	#reset
rand			and	data, #$FF			' isolate byte
look_ret
receive_ack_ret
receive_ret
			ret
'
'
' Wait for clock/data to be in required state(s)
'
wait_c0			mov	wcond, c0			' (wait until clock low)
wait			mov	y, tenms			' set timeout to 10ms
wloop			movs	napshr, #18			' nap ~4us
			call	#nap
_c3			test	cmask, ina		wc	' check required state(s)
_d4			test	dmask, ina		wz	' loop until got state(s) or timeout
wcond	if_never	djnz	y, #wloop			' (replaced with c0/c1/c0d0/c1d1)
			tjz	y, #reset			' if timeout, reset keyboard
wait_ret
wait_c0_ret
			ret


c0	if_c		djnz	y, #wloop			' (if_never replacements)
c1	if_nc		djnz	y, #wloop
c0d0	if_c_or_nz	djnz	y, #wloop
c1d1	if_nc_or_z	djnz	y, #wloop
'
'
' Nap
'
nap			rdlong	t, #0				' get clkfreq
napshr			shr	t, #18/16/13			' shr scales time
			min	t, #3				' ensure waitcnt won't snag
			add	t, cnt				' add cnt to time
			waitcnt t, #0				' wait until time elapses (nap)
nap_ret
			ret
'
'
' Initialized data
'
'
dlsb			long	1 << 9
tenms			long	10_000 / 4
'
'***************************************************************************
'*   w/o SHIFT                             with SHIFT
'*   +-------------------------------+     +-------------------------------+
'*   | 0   1   2   3   4   5   6   7 |     | 0   1   2   3   4   5   6   7 |
'*+--+---+---+---+---+---+---+---+---+  +--+---+---+---+---+---+---+---+---+
'*|0 | @ | A | B | C | D | E | F | G |  |0 | ` | a | b | c | d | e | f | g |
'*|  +---+---+---+---+---+---+---+---+  |  +---+---+---+---+---+---+---+---+
'*|1 | H | I | J | K | L | M | N | O |  |1 | h | i | j | k | l | m | n | o |
'*|  +---+---+---+---+---+---+---+---+  |  +---+---+---+---+---+---+---+---+
'*|2 | P | Q | R | S | T | U | V | W |  |2 | p | q | r | s | t | u | v | w |
'*|  +---+---+---+---+---+---+---+---+  |  +---+---+---+---+---+---+---+---+
'*|3 | X | Y | Z | [ | \ | ] | ^ | _ |  |3 | x | y | z | { | | | } | ~ |   |
'*|  +---+---+---+---+---+---+---+---+  |  +---+---+---+---+---+---+---+---+
'*|4 | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 |  |4 |   | ! | " | # | $ | % | & | ' |
'*|  +---+---+---+---+---+---+---+---+  |  +---+---+---+---+---+---+---+---+
'*|5 | 8 | 9 | : | ; | , | - | . | / |  |5 | ( | ) | * | + | < | = | > | ? |
'*|  +---+---+---+---+---+---+---+---+  |  +---+---+---+---+---+---+---+---+
'*|6 |ENT|CLR|BRK|UP |DN |LFT|RGT|SPC|  |6 |ENT|CLR|BRK|UP |DN |LFT|RGT|SPC|
'*|  +---+---+---+---+---+---+---+---+  |  +---+---+---+---+---+---+---+---+
'*|7 |SHF|ALT|PUP|PDN|INS|DEL|CTL|END|  |7 |SHF|ALT|PUP|PDN|INS|DEL|CTL|END|
'*+--+---+---+---+---+---+---+---+---+  +--+---+---+---+---+---+---+---+---+
'* not visible to TRS80
'*+--+---+---+---+---+---+---+---+---+
'*|8 | F1| F2| F3| F4| F5| F6| F7| F8|
'*+--+---+---+---+---+---+---+---+---+
'*|9 | F9|F10|F11|F12|   |   |   |   |
'*+--+---+---+---+---+---+---+---+---+
'*|10|WiL|WiR|App|Pow|Slp|Wak|Prt|   |
'*+--+---+---+---+---+---+---+---+---+
'***************************************************************************
'
'
' Lookup table
'				TRS80	scan	extkey	regkey	()=keypad
'
table			word	$ffff	'00
			word	$ff48	'01		F9
			word	$ffff	'02
			word	$ff44	'03		F5
			word	$ff42	'04		F3
			word	$ff40	'05		F1
			word	$ff41	'06		F2
			word	$ff4b	'07		F12
			word	$ffff	'08
			word	$ff49	'09		F10
			word	$ff47	'0A		F8
			word	$ff46	'0B		F6
			word	$ff43	'0C		F4
			word	$ff36	'0D		Tab
			word	$ff00	'0E		`
			word	$ffff	'0F
			word	$ffff	'10
			word	$3939	'11	Alt-R	Alt-L
			word	$ff38	'12		Shift-L
			word	$ffff	'13
			word	$3e3e	'14	Ctrl-R	Ctrl-L
			word	$ff11	'15		q
			word	$ff21	'16		1
			word	$ffff	'17
			word	$ffff	'18
			word	$ffff	'19
			word	$ff19	'35		y
'			word	$ff1a	'1A		z
			word	$ff13	'1B		s
			word	$ff01	'1C		a
			word	$ff17	'1D		w
			word	$ff22	'1E		2
			word	$50ff	'1F	Win-L
			word	$ffff	'20
			word	$ff03	'21		c
			word	$ff18	'22		x
			word	$ff04	'23		d
			word	$ff05	'24		e
			word	$ff24	'25		4
			word	$ff23	'26		3
			word	$51ff	'27	Win-R
			word	$ffff	'28
			word	$ff37	'29		Space
			word	$ff16	'2A		v
			word	$ff06	'2B		f
			word	$ff14	'2C		t
			word	$ff12	'2D		r
			word	$ff25	'2E		5
			word	$52ff	'2F	Apps
			word	$ffff	'30
			word	$ff0E	'31		n
			word	$ff02	'32		b
			word	$ff08	'33		h
			word	$ff07	'34		g
			word	$ff1a	'1A		z
'			word	$ff19	'35		y
			word	$ff26	'36		6
			word	$53ff	'37	Power
			word	$ffff	'38
			word	$ffff	'39
			word	$ff0d	'3A		m
			word	$ff0a	'3B		j
			word	$ff15	'3C		u
			word	$ff27	'3D		7
			word	$ff28	'3E		8
			word	$54ff	'3F	Sleep
			word	$ffff	'40
			word	$ff2C	'41		,
			word	$ff0B	'42		k
			word	$ff09	'43		i
			word	$ff0F	'44		o
			word	$ff20	'45		0
			word	$ff29	'46		9
			word	$ffff	'47
			word	$ffff	'48
			word	$ff2e	'49		.
			word	$2f2f	'4A	(/)	/
			word	$ff0C	'4B		l
			word	$ff2b	'4C		;
			word	$ff10	'4D		p
			word	$ff2d	'4E		-
			word	$ffff	'4F
			word	$ffff	'50
			word	$ffff	'51
			word	$ff2e	'52		'
			word	$ffff	'53
			word	$ff1b	'54		[
			word	$ff2d	'55		=
			word	$ffff	'56
			word	$ffff	'57
			word	$ffde	'58		CapsLock
			word	$ff38	'59		Shift-R
			word	$3030	'5A	(Enter) Enter
			word	$ff1d	'5B		]
			word	$ffff	'5C
			word	$ff1c	'5D		\
			word	$55ff	'5E	WakeUp
			word	$ffff	'5F
			word	$ffff	'60
			word	$ffff	'61
			word	$ffff	'62
			word	$ffff	'63
			word	$ffff	'64
			word	$ffff	'65
			word	$ff35	'66		BackSpace
			word	$ffff	'67
			word	$ffff	'68
			word	$3f21	'69	End	(1)
			word	$ffff	'6A
			word	$3524	'6B	Left	(4)
			word	$3127	'6C	Home	(7)
			word	$ffff	'6D
			word	$ffff	'6E
			word	$ffff	'6F
			word	$3c20	'70	Insert	(0)
			word	$3d2e	'71	Delete	(.)
			word	$3422	'72	Down	(2)
			word	$0025	'73		(5)
			word	$3626	'74	Right	(6)
			word	$3328	'75	Up	(8)
			word	$ff32	'76		Esc
			word	$ffdf	'77		NumLock
			word	$ff49	'78		F11
			word	$ff2b	'79		(+)
			word	$3b23	'7A	PageDn	(3)
			word	$ff2d	'7B		(-)
			word	$562a	'7C	PrScr	(*)
			word	$3a29	'7D	PageUp	(9)
			word	$ffdd	'7E		ScrLock
			word	$ffff	'7F
			word	$ffff	'80
			word	$ffff	'81
			word	$ffff	'82
			word	$ff46	'83		F7
'
'
' Uninitialized data
'
dmask			res	1
cmask			res	1
stat			res	1
data			res	1
x			res	1
y			res	1
t			res	1

_states			res	8	'write-only
_present		res	1	'write-only
_dpin			res	1	'read-only at start
_cpin			res	1	'read-only at start
_locks			res	1	'read-only at start
_auto			res	1	'read-only at start

{{

+------------------------------------------------------------------------------------------------------------------------------+
|                                                   TERMS OF USE: MIT License                                                  | +------------------------------------------------------------------------------------------------------------------------------+
|Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    | |files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    |
|modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software|
|is furnished to do so, subject to the following conditions:                                                                   |
|                                                                                                                              |
|The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.|
|                                                                                                                              |
|THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          |
|WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         |
|COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   |
|ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         |
+------------------------------------------------------------------------------------------------------------------------------+
}}
