{{      Bellatrix-Code
┌──────────────────────────────────────────────────────────────────────────────────────────────────────┐
│ Autor: Ingo Kripahle                                                                                 │
│ Copyright (c) 2012 Ingo Kripahle                                                                     │
│ See end of file for terms of use.                                                                    │
│ Die Nutzungsbedingungen befinden sich am Ende der Datei                                              │
└──────────────────────────────────────────────────────────────────────────────────────────────────────┘

Informationen   : hive-project.de
Kontakt         : drohne235@gmail.com
System          : mental
Name            : VGA-Text-Treiber 1024x768 Pixel, 64x24 Zeichen
Chip            : Bellatrix
Typ             : Treiber
Funktion        :

                  - Standard VGA-Text- und Tastatur-Treiber
                  - Eingabezeile fÃ¼r m-core
                  - Zahleformatierung
                  - Screeneditor

Komponenten     : VGA 1024x768 Tile Driver v0.9   Chip Gracey        MIT
                  PS/2 Keyboard Driver v1.0.1     Chip Gracey, ogg   MIT

COG's           : MANAGMENT     1 COG
                  VGA           2 COG's
                  KEYB          1 COG
                  Task          1 COG
                  -------------------
                                5 COG's

Logbuch         :

31-10-2012-dr235  - textmodus eingefÃ¼gt



Notizen:

ACHTUNG: row ist nicht die Zeilenposition, da zwei tiles untereinander ein zeichen
bilden. vielmehr ist die reale zeilenposition row/2.


}}

CON

_CLKMODE     = XTAL1 + PLL16X
_XINFREQ     = 5_000_000


COLS         = 64
ROWS         = 48
TILES        = COLS * ROWS
RESX         = 1024
RESY         = 768
COLORANZ     = 8
TABANZ       = 8
OUTCUR       = $0F                                     'cursor ausgabe
INPCUR       = $0E                                     'cursor eingabe
RDYCUR       = $07
SPACETILE    = $8000 + $20 << 6

VGA_BASPORT  = gc#bel_vgabase                          'vga startport
VGA_RESX     = COLS * 16                               'vga anzahl pixel x
VGA_RESY     = ROWS * 16                               'vga anzahl pixel y
KEYB_DPORT   = gc#bel_keybd                            'tastatur datenport
KEYB_CPORT   = gc#bel_keybc                            'tastatur taktport

'          hbeat   --------+
'          clk     -------+|
'          /wr     ------+||
'          /hs     -----+||| +------------------------- /cs
'                       |||| |                 -------- d0..d7
DB_IN            = %00001001_00000000_00000000_00000000 'maske: dbus-eingabe
DB_OUT           = %00001001_00000000_00000000_11111111 'maske: dbus-ausgabe

M1               = %00000010_00000000_00000000_00000000
M2               = %00000010_10000000_00000000_00000000 'busclk=1? & /cs=0?

M3               = %00000000_00000000_00000000_00000000
M4               = %00000010_00000000_00000000_00000000 'busclk=0?

'farben fÃ¼r blinkenden cursor

C1A     = $30003000       '+gr/bk
C1B     = $30300000
C2A     = $00300030       '-gr/bk
C2B     = $00003030

OBJ

  vga        : "m-bel-vga"
  key        : "m-bel-key"
  num        : "m-glob-num"
  gc         : "m-glob-con"

VAR

  long  col, row, color                                 'spalten-, zeilenposition und zeichenfarbe
  long  array[TILES/2]                                  'bildschirmpuffer
  byte  cursor                                          'cursorzeichen
  byte  curstat                                         'cursorstatus 1 = ein
  byte  sline                                           'startzeile des scrollfensters (0 = 1. zeile)
  byte  eline                                           'endzeile des scrollfensters (0 = 1. zeile)
  byte  tab[TABANZ]                                     'tabulatorpositionen

  long  plen                                            'lÃ¤nge datenblock loader
  byte  proghdr[16]                                     'puffer fÃ¼r objektkopf

  byte  ctag                                            'aktives colortag
  byte  base                                            'zahlenbasis
  byte  error                                           'fehlercode

long    taskstack[32]

CON ''------------------------------------------------- BELLATRIX

PUB main | zeichen,a,b,keycode                          'chip: managmentcode
''funktionsgruppe               : chip
''funktion                      : kommandointerpreter
''eingabe                       : -
''ausgabe                       : -

  init_subsysteme                                       'bus/vga/keyboard/maus initialisieren
  repeat
    zeichen := bus_getchar                              '1. zeichen empfangen
    if zeichen                                          ' > 0
      print_char(zeichen)
    else
      zeichen := bus_getchar                            '2. zeichen kommando empfangen
      case zeichen
        gc#bel_key_stat:        bus_putchar(key.gotkey) 'tastaturstatus senden
        gc#bel_key_code:        keycode := key.key      'tastencode senden
                                bus_putchar(keycode)
        gc#bel_key_spec:        bus_putchar(keycode>>8) 'statustasten ($100..$1FF) abfragen
        gc#bel_key_wait:        keycode := key.getkey   'auf tastencode warten
                                bus_putchar(keycode)
        gc#bel_pchar:           pchar(bus_getchar)      'zeichen ohne steuerzeichen augeben
        gc#bel_setx:            col := bus_getchar      'x-position setzen
        gc#bel_sety:            row := bus_getchar*2    'y-position setzen
        gc#bel_getx:            bus_putchar(col)        'x-position abfragen
        gc#bel_gety:            bus_putchar(row/2)      'y-position abfragen
        gc#bel_color:           color := bus_getchar    'farbe setzen
        gc#bel_sline:           sline := bus_getchar*2  'startzeile scrollbereich
        gc#bel_eline:           eline := bus_getchar*2  'endzeile scrollbereich
        gc#bel_settab:          a := bus_getchar        'tabulatorposition setzen
                                b := bus_getchar
                                if a =< (TABANZ-1)
                                  tab[a] := b
'       ----------------------------------------------  M-FUNKTIONEN
        gc#bel_m_parse:         m_parse                 'nÃ¤chstes token von eingabezeile parsen
        gc#bel_m_setbase:       m_setbase               'base setzen
        gc#bel_m_dot:           m_dot                   'formatierte ausgabe eines zahlenwertes
        gc#bel_m_error:         m_error                 'm fehlermeldung
'       ----------------------------------------------  SCREENEDITOR
        gc#bel_scr_edit:        scr_edit                'screeneditor
        gc#bel_scr_put:         scr_put                 'screen empfangen
        gc#bel_scr_get:         scr_get                 'screen senden
        gc#bel_scr_setnr:       scr_setnr               'screennummer setzen
'       ----------------------------------------------  CHIP-MANAGMENT
        gc#bel_mgr_setcolor:    mgr_setcolor            'farbregister setzen
        gc#bel_mgr_load:        mgr_load                'neuen bellatrix-code laden
        gc#bel_reboot:          reboot                  'bellatrix neu starten

PRI init_subsysteme|i                                   'chip: initialisierung des bellatrix-chips
''funktionsgruppe               : chip
''funktion                      : - initialisierung des businterface
''                              : - vga & keyboard-treiber starten
''eingabe                       : -
''ausgabe                       : -

  dira := db_in                                         'datenbus auf eingabe schalten
  outa[gc#bus_hs] := 1                                  'handshake inaktiv

  print_char(gc#bel_cls)                                'bildschirm lÃ¶schen

  key.start(keyb_dport, keyb_cport)                     'tastaturport starten
  vga.start(vga_basport,@array,@vgacolors,0,0,0)        'vga-treiber starten
  cursor := OUTCUR                                      'cursorzeichen setzen
  curstat := 1                                          'cursor anschalten
  sline := 0                                            'startzeile des scrollbereichs setzen
  eline := rows-6                                       'endzeile des scrollbereichs setzen
  repeat i from 0 to TABANZ-1                           'tabulatoren setzen
    tab[i] := i * 4

  'eingabezeile/editor konfigurieren

  tibempty   := 0
  tibpos     := 0
  base       := 10
  error      := 0
  scr_edline := 0
  scr_invmode:= 0
  scr_clear                                             'screenpuffer lÃ¶schen

  'cog fÃ¼r hintergrundfunktionen starten

  cognew(screentask,@taskstack)

CON ''------------------------------------------------- HINTERGRUNDFUNKTIONEN

PRI screentask
' jo, momentan leisten wir uns den luxus mit einem kompletten
' 32 bit core den cursor blinken zu lassen  :)

  repeat
    long[@cursordat1] := C1A
    long[@cursordat2] := C1B
    waitcnt(cnt+clkfreq/8)
    if curstat
      long[@cursordat1] := C2A
      long[@cursordat2] := C2B
      waitcnt(cnt+clkfreq*2)

CON ''------------------------------------------------- BUS-FUNKTIONEN

PRI bus_putchar(zeichen)                                'bus: ein byte an regnatix senden
''funktionsgruppe               : bus
''funktion                      : ein byte an regnatix senden
''eingabe                       : byte
''ausgabe                       : -

  waitpeq(M1,M2,0)                                      'busclk=1? & prop2=0?
  dira := db_out                                        'datenbus auf ausgabe stellen
  outa[7..0] := zeichen                                 'daten ausgeben
  outa[gc#bus_hs] := 0                                  'daten gÃ¼ltig
  waitpeq(M3,M4,0)                                      'busclk=0?
  dira := db_in                                         'bus freigeben
  outa[gc#bus_hs] := 1                                  'daten ungÃ¼ltig

PRI bus_getchar : zeichen                               'bus: ein byte von regnatix empfangen
''funktionsgruppe               : bus
''funktion                      : ein byte von regnatix empfangen
''eingabe                       : -
''ausgabe                       : byte

   waitpeq(M1,M2,0)                                     'busclk=1? & prop2=0?
   zeichen := ina[7..0]                                 'daten einlesen
   outa[gc#bus_hs] := 0                                 'daten quittieren
   waitpeq(M3,M4,0)                                     'busclk=0?
   outa[gc#bus_hs] := 1

PRI sub_putlong(wert)                                   'bus: long senden
''funktionsgruppe               : bus
''funktion                      : protokoll um einen long-wert an regnatix zu senden
''eingabe                       : 32bit wert der gesendet werden soll
''ausgabe                       : -
''busprotokoll                  : [put.byte1][put.byte2][put.byte3][put.byte4]
''                              : [  hsb    ][         ][         ][   lsb   ]

   bus_putchar(wert >> 24)                              '32bit wert senden hsb/lsb
   bus_putchar(wert >> 16)
   bus_putchar(wert >> 8)
   bus_putchar(wert)

PRI sub_getlong:wert                                    'bus: long empfangen
''funktionsgruppe               : bus
''funktion                      : protokoll um einen long-wert von regnatix zu empfangen
''eingabe                       : -
''ausgabe                       : 32bit-wert der empfangen wurde
''busprotokoll                  : [get.byte1][get.byte2][get.byte3][get.byte4]
''                              : [  hsb    ][         ][         ][   lsb   ]

  wert :=        bus_getchar << 24                      '32 bit empfangen hsb/lsb
  wert := wert + bus_getchar << 16
  wert := wert + bus_getchar << 8
  wert := wert + bus_getchar

PRI sub_putstr(strptr)|len,i                            'sub: string senden
''funktionsgruppe               : sub
''funktion                      : subprotokoll um einen string an regnatix zu senden
''eingabe                       : strptr - zeiger auf einen string (0-term)
''ausgabe                       : -
''busprotokoll                  : [put.len][put.byte(1)]..[put.byte(len)]
''                              : len - lÃ¤nge des dateinamens

  len := strsize(strptr)
  bus_putchar(len)
  repeat i from 0 to len - 1                            'string Ã¼bertragen
    bus_putchar(byte[strptr][i])

PRI sub_mtoken                                          'sub: m-token senden

  bus_putchar(tok_tag)
  sub_putstr(@tok)

PRI sub_mnumber                                         'sub: m-wert senden

  bus_putchar(tok_tag)
  sub_putlong(m_number(@tok))

CON ''------------------------------------------------- M-FUNKTIONEN

CON                                                     'm: konstanten

  inputline     = rows - 6

PRI m_parse | i                                         'm: liefert geparste token

  repeat
    ifnot tibempty                                      'wenn tib leer,
      tib_input(inputline,1)                            'neue zeile eingeben.
    m_next                                              'nÃ¤chstes token holen
  while (tok_tag == gc#m_c_remark) or (strsize(@tok) == 0) 'kommentare & leere token ausblenden

  case tok_tag
    gc#m_c_tag1: sub_mtoken          'wort ausfÃ¼hren
    gc#m_c_tag2: sub_mtoken          'wort definieren
    gc#m_c_tag3: sub_mtoken          'wort compilieren
    gc#m_c_tag4: sub_mnumber         'zahl
    gc#m_c_tag5: sub_mnumber         'zahl literal
    gc#m_c_tag6: sub_mtoken          'string
    gc#m_c_tag7: sub_mtoken          'string literal
    gc#m_c_tag8: sub_mtoken          'data


PRI m_next: flag | i,pos                                'm: parst nÃ¤chstes token aus eingabe

  'tokenpuffer lÃ¶schen
  repeat i from 0 to TIBMAX+2
    tok[i] := 0

  pos := 0
  'fÃ¼hrende tags ausblenden, tag fÃ¼r parser speichern
  repeat until (tib[tok_pos] > gc#m_c_max) or (tib[tok_pos] == 0) 'TIB_SP
    tok_pos++
  tok_tag := tib[tok_pos - 1]

  'lÃ¤nge ermitteln, token kopieren
  repeat until (tib[tok_pos] < TIB_SP) or (tib[tok_pos] == 0)
    tok[pos++] := tib[tok_pos++]

  tibempty := tib[tok_pos] ' letztes zeichen als flag Ã¼bergeben
  return tibempty


PRI m_setbase                                           'm: base-variable setzen

  base := bus_getchar

PRI m_number(strptr): n                                 'm: formatierte zahleneingabe

  case tok[0]
    "%": return num.FromStr(@tok,num#BIN)
    "$": return num.FromStr(@tok,num#HEX)

  case base
    02: return num.FromStr(@tok,num#BIN)
    10: return num.FromStr(@tok,num#DEC)
    16: return num.FromStr(@tok,num#HEX)

PRI m_dot                                               'm: formatierte zahlenausgabe

  case base
    02: print_bin(sub_getlong,bus_getchar)
    10: print_dec(sub_getlong,bus_getchar)
    16: print_hex(sub_getlong,bus_getchar)

PRI m_error                                             'm: fehlerausgabe

  error := bus_getchar
  if error
    tib_error
    tib_clear

CON ''------------------------------------------------- SCREENEDITOR

CON                                                     'scr: konstanten

  SCRLINEMAX    = 64
  SCRLINES      = 16
  SCRSIZE       = SCRLINEMAX*SCRLINES
  CMDLINE       = 18
  COPYLINE      = 17
  STATLINE      = 21
  INFOLINE        = 22

VAR                                                     'scr: variablen

  byte  scr0[SCRSIZE] 'tib
  byte  scr_edline
  long  scr_screen
  long  scr_max
  byte  scr_mode
  byte  scr_invmode

PRI scr_edit | k,s                                      'scr: editor

  scr_pushcur
  print_char(gc#bel_cls)        'bildschirm lÃ¶schen
  tib_clear                     'eingabepuffer lÃ¶schen
  tibctag := gc#M_C_TAG9        'tag auf kommentar setzen
  scr_mode := 0                 'colortag-modus
  scr_disp                      'editorscreen ausgeben
  scr_popline(scr_edline)       'zeile puffer --> tib
  scr_tibdisp(scr_edline,1)     'zeile anzeigen

  repeat
    k := key.getkey             'keyboard abfragen
    s := k >> 8                 ' spezifikation
    k := k & $FF                ' keycode
    scr_key(k,s)
    scr_tibdisp(scr_edline,1)   'zeile aktualisieren
    scr_copy                    'kopierpuffer aktualisieren
    scr_status                  'status aktualisieren
  until k == TIB_ES

  if scr_invmode
    scr_inverter                  'modus fÃ¼r invertierte farben rÃ¼ckstellen
  scr_pushline(scr_edline)      'zeile tib --> puffer
'  print_char(gc#bel_cls)       'bildschirm lÃ¶schen
  scr_popcur

PRI scr_inputchar(stringptr):char|colback,rowback       'scr: eingabe eines zeichens

  colback := col
  rowback := row
  color := 0
  row := INFOLINE * 2
  col := 0
  print_str(stringptr)
  char := key.getkey
  print_char(char)
  col := colback
  row := rowback

PRI scr_pushcur                                         'scr: log-cursor sichern
  tibeline := eline
  eline := rows-6                                       'endzeile des scrollbereichs setzen


PRI scr_popcur                                          'scr: log-cursor wiederherstelle
  row := CMDLINE * 2
  col := 0
  eline := tibeline

PRI scr_get | i                                         'scr: screen --> core

  i := 0
  repeat SCRSIZE
    if scr0[i]                  ' 0 --> space
      bus_putchar(scr0[i])
    else
      bus_putchar(gc#m_c_remark) '(32)
    i++

PRI scr_put | i                                         'scr: core --> screen

  i := 0
  repeat SCRSIZE
    scr0[i++] := bus_getchar

PRI scr_setnr                                           'scr: screennummer im status setzen

  scr_screen := sub_getlong
  scr_max    := sub_getlong

PRI scr_status                                          'scr: anzeige statusbereich

  row := STATLINE * 2
  col := 0
  color := tibctag - gc#m_c_tag1
  tib_printtag(tibctag)
  color := gc#c_normal
  print_str(string("[F10:CPY|11:INS|12:CLR]"))
  print_str(string(" SCR:"))
  print_dec(scr_screen,8)
  print_str(string(" L: "))
  print_dec(scr_edline,2)
  print_str(string(" : "))
  print_dec(tibpos,2)
  color := gc#c_normal

PRI scr_copy | i                                        'scr: anzeige kopierpuffer

  color := 0
  row := COPYLINE * 2
  col := 0
  repeat i from 0 to TIBMAX-1
    print_char(tibcopy[i])

PRI scr_disp | i                                        'scr: editorscreen aktualisieren

  i := 0
  repeat SCRLINES
    scr_line(i++)
  color := gc#c_normal
  repeat COLS
    print_char("-")
  scr_copy                      'kopierpuffer anzeigen
  scr_status                    'status anzeigen

PRI scr_line(line) | i                                  'scr: ausgabe einer editorzeile an cursorposition

  tibpos1 := line * 2
  i := line * SCRLINEMAX
  repeat SCRLINEMAX   'puffer nach cursorposition
    print_char(scr0[i++])

PRI scr_tibdisp(line,mode) | i,j,colback                'scr: tib --> display

' line  - bildschirmzeile, an welcher der tib ausgegeben werden soll
' mode  - 1 = mit cursor

  tibpos1 := line * 2
  tib_pos1
  i := 0
  '----------------------------------------------------- zeile vor cursor
  if tibpos
    repeat i from 0 to tibpos-1
      print_char(tib[i])
  '----------------------------------------------------- cursor
  if mode
    colback := color
    color := gc#c_cursor
    if tib[i] > TIB_SP          'space oder leerzeichen?
      print_char(tib[i++])      'normales zeichen einfÃ¤rben
      color := colback
    else
      print_char(TIB_SP)      'leerzeichen einfÃ¤rben
      i++
      if scr_mode == 1
        color := colback
  '----------------------------------------------------- zeile nach cursor
  if tibpos < TIBMAX-1
    repeat j from i to TIBMAX-1   'puffer nach cursorposition
      print_char(tib[j])

PRI scr_key(k,s)                                        'scr: zeicheneingabe

  if s==2                       'ctrl-taste?
    case k
      "a": scr_mode := 1        'ascii-modus
           tib_delcopy
      "c": scr_mode := 0        'color-modus
           tib_delcopy
      "n": if scr_inputchar(string(" Screen löschen? [j|n] : ")) == "j"
             scr_clear
             tib_fill
             print_char(gc#bel_cls)        'bildschirm lÃ¶schen
             scr_disp
             case scr_mode
               0: tibpos := 1
               1: tibpos := 0
             scr_edline := 0
           else
             print_char(gc#bel_cls)        'bildschirm lÃ¶schen
             scr_disp
      "i": scr_inverter         'colortags mit speziellem handling von space werden invertiert
    return

  case scr_mode
    0: scr_ckey(k)
    1: scr_tkey(k)

PRI scr_inverter                                        'scr: darstellungsmodus sring invertieren

  if scr_invmode
    scr_invmode := 0
    long[@col_string1] := $E000E000
    long[@col_string2] := $E0E00000
    long[@col_stringlit1] := $50005000
    long[@col_stringlit2] := $50500000
  else
    scr_invmode := 1
    long[@col_string1] := $00D000D0
    long[@col_string2] := $0000D0D0
    long[@col_stringlit1] := $00E000E0
    long[@col_stringlit2] := $0000E0E0

{

marken

col_string
col_stringlit
col_remark

normale farben

  long $D0FCD0FC       'orange - string
  long $D0D0FCFC

  long $E0FCE0FC       'gelb - string literal
  long $E0E0FCFC

  long $54FC54FC       'grau - kommentar
  long $5454FCFC

inverse farben

  long $00D000D0       'schwarz auf orange
  long $0000D0D0

  long $00E000E0       'schwarz auf gelb
  long $0000E0E0

  long $00540054       'schwarz auf grau
  long $00005454

}

PRI scr_tkey(k)                                         'scr: zeicheneingabe - text-modus

  case k

    TIB_F10: tib_copy
    TIB_F11: tib_past
    TIB_F12: tib_fill

    TIB_LC:                                             ' < cursor
      if tibpos > 0
        tibpos--
    TIB_RC:                                             ' cursor>
      if tibpos < TIBMAX-1
        tibpos++
    TIB_UC:                                             ' cursor up
      scr_pushline(scr_edline)
      scr_tibdisp(scr_edline,0)
      if scr_edline < (SCRLINES - 1)
        scr_edline++
      scr_popline(scr_edline)
    TIB_DC:                                             ' cursor down
      scr_pushline(scr_edline)
      scr_tibdisp(scr_edline,0)
      if scr_edline
        scr_edline--
      scr_popline(scr_edline)
    TIB_BS:                                             ' backspace
      if tibpos > 0
        bytemove(@tib+tibpos-1, @tib+tibpos, TIBMAX+1-tibpos)
        tibpos--
    TIB_P1: tibpos := 0
    TIB_TB: tib_tabpos
    TIB_CR:
      scr_pushline(scr_edline)
      scr_tibdisp(scr_edline,0)
      if scr_edline < (SCRLINES - 1)
        scr_edline++
      scr_popline(scr_edline)
      tibpos := 0
    $20..$FF: tib_setchar(k)

PRI scr_ckey(k)                                         'scr: zeicheneingabe - color-modus

  case k

    TIB_F1:  tib_setLastTag(gc#m_c_tag1)
    TIB_F2:  tib_setLastTag(gc#m_c_tag2)
    TIB_F3:  tib_setLastTag(gc#m_c_tag3)
    TIB_F4:  tib_setLastTag(gc#m_c_tag4)
    TIB_F5:  tib_setLastTag(gc#m_c_tag5)
    TIB_F6:  tib_setLastTag(gc#m_c_tag6)
    TIB_F7:  tib_setLastTag(gc#m_c_tag7)
    TIB_F8:  tib_setLastTag(gc#m_c_tag8)
    TIB_F9:  tib_setLastTag(gc#m_c_tag9)
    TIB_F10: tib_copy
    TIB_F11: tib_past
    TIB_F12: tib_fill

    TIB_LC:                                             ' < cursor
      if tibpos > 1
        tibpos--
    TIB_RC:                                             ' cursor>
      if tibpos < TIBMAX-1
        tibpos++
    TIB_UC:                                             ' cursor up
      scr_pushline(scr_edline)
      scr_tibdisp(scr_edline,0)
      if scr_edline < (SCRLINES - 1)
        scr_edline++
      scr_popline(scr_edline)
    TIB_DC:                                             ' cursor down
      scr_pushline(scr_edline)
      scr_tibdisp(scr_edline,0)
      if scr_edline
        scr_edline--
      scr_popline(scr_edline)
    TIB_BS:                                             ' backspace
      if tibpos > 1
        bytemove(@tib+tibpos-1, @tib+tibpos, TIBMAX+1-tibpos)
        tibpos--
    TIB_P1: tibpos := 1
    TIB_TB: tib_tabpos
    TIB_SP: tib_setspace
    TIB_CR:
      scr_pushline(scr_edline)
      scr_tibdisp(scr_edline,0)
      if scr_edline < (SCRLINES - 1)
        scr_edline++
      scr_popline(scr_edline)
      tibpos := 1
    $21..$FF: tib_setchar(k)

PRI scr_popline(line) | i                               'scr: zeile puffer --> tib

  bytemove(@tib, @scr0 + (line * SCRLINEMAX), SCRLINEMAX]

PRI scr_pushline(line) | i                              'scr: zeile tib --> puffer

  bytemove(@scr0 + (line * SCRLINEMAX), @tib, SCRLINEMAX]

PRI scr_clear | i,n

  case scr_mode
    0: n := gc#M_C_TAG9
    1: n := " "
  i := 0
  repeat SCRSIZE
    scr0[i++] := n


CON ''------------------------------------------------- EINGABEZEILE

CON                                                     'tib: konstanten

  TIBMAX        = 64
  TIB_CR        = $0D
  TIB_BS        = $C8
  TIB_SP        = $20
  TIB_LC        = $02
  TIB_RC        = $03
  TIB_DC        = $04
  TIB_UC        = $05
  TIB_P1        = $06
  TIB_TB        = $09
  TIB_ES        = $1B

  TIB_F1        = $D0
  TIB_F2        = $D1
  TIB_F3        = $D2
  TIB_F4        = $D3
  TIB_F5        = $D4
  TIB_F6        = $D5
  TIB_F7        = $D6
  TIB_F8        = $D7
  TIB_F9        = $D8
  TIB_F10       = $D9
  TIB_F11       = $DA
  TIB_F12       = $DB
  TIB_CHAR      = $21
{
  C_TAG1        = gc#m_c_tag1           'wort ausfÃ¼hren
  C_TAG2        = gc#m_c_tag2           'wort definieren
  C_TAG3        = gc#m_c_tag3           'wort compilieren
  C_TAG4        = gc#m_c_tag4           'zahl>stack
  C_TAG5        = gc#m_c_tag5           'zahl compilieren
  C_TAG6        = gc#m_c_tag6           'string
  C_TAG7        = gc#m_c_tag7           'kommentar
  C_TAG8        = gc#m_c_tag8           'cursor
  C_REM         = gc#m_c_tag7
}

VAR                                                     'tib: variablen

  byte  tib[TIBMAX+2]           'tib
  byte  tibcopy[TIBMAX+2]       'tib-kopierpuffer
  word  tibempty                'flag 0 = tib leer
  word  tibpos                  'position im tib
  word  tibpos1                 'startposition eingabezeile
  byte  tibx
  byte  tiby
  byte  tibeline
  byte  tibcur
  byte  tibctag

  byte  tok[TIBMAX+2] 'token
  byte  tok_tag
  byte  tok_pos

PRI tib_input(line,mode) | k                            'tib: eingabe einer zeile

  tibpos1 := line
  color := gc#c_attention
  print_str(string(" ok "))
  print_char($0D)
  tib_pushcur
  tib_clear
  tib_disp(mode)
  repeat
    k := key.getkey & $FF
    tib_key(k)
    tib_disp(mode)
  until k == TIB_CR
  tib_popcur
  print_str(@tib)
  color := gc#c_attention
  print_char(" ")
  pchar("▶")
  print_char(" ")
  color := gc#c_normal


PRI tib_pushcur                                         'tib: log-cursor sichern
  tibx := col
  tiby := row
  tibeline := eline
  eline := rows-6                                       'endzeile des scrollbereichs setzen

PRI tib_popcur                                          'tib: log-cursor wiederherstelle
  col := tibx
  row := tiby
  eline := tibeline

PRI tib_key(k)                                          'tib: zeicheneingabe

  case k

    TIB_F1:  tib_setLastTag(gc#m_c_tag1)
    TIB_F2:  tib_setLastTag(gc#m_c_tag2)
    TIB_F3:  tib_setLastTag(gc#m_c_tag3)
    TIB_F4:  tib_setLastTag(gc#m_c_tag4)
    TIB_F5:  tib_setLastTag(gc#m_c_tag5)
    TIB_F6:  tib_setLastTag(gc#m_c_tag6)
    TIB_F7:  tib_setLastTag(gc#m_c_tag7)
    TIB_F8:  tib_setLastTag(gc#m_c_tag8)
    TIB_F9:  tib_setLastTag(gc#m_c_tag9)
    TIB_F10:
    TIB_F11:
    TIB_F12:


    TIB_LC:                                             ' < cursor
      if tibpos > 1
        tibpos--
    TIB_RC:                                             ' cursor>
      if tibpos < TIBMAX-1
        tibpos++
    TIB_BS:                                             ' backspace
      if tibpos > 1
        bytemove(@tib+tibpos-1, @tib+tibpos, TIBMAX+1-tibpos)
        tibpos--
    TIB_P1: tibpos := 1
    TIB_TB: tib_tabpos
    TIB_SP: tib_setspace
    $21..$FF: tib_setchar(k)

PRI tib_setspace                                        'tib: leerzeichen einfÃ¼gen

  case tibctag
    gc#m_c_string:    tib_setchar($20)     'strings bekommen leerzeichen
    gc#m_c_stringlit: tib_setchar($20)     'strings bekommen leerzeichen
    gc#m_c_remark:    tib_setchar($20)     'kommentare bekommen leerzeichen
    other:  tib_setchar(tibctag) 'der rest bekommt das letzte colortag

PRI tib_tabpos | i                                      'tib: springe zum nÃ¤chsten token

  repeat i from tibpos to TIBMAX-1
    case tib[i]
      gc#m_c_tag1..gc#m_c_tag7:           ' colortag gefunden
        tibpos := i + 1
        return
  if i := TIBMAX-1  ' ist ende erreicht, wieder bei pos1 anfangen
    tibpos := 1

PRI tib_setLastTag(k) | i                               'tib: setzt colortag vom token

  tibctag := k
  repeat i from tibpos-1 to 0
    case tib[i]
      0:                                                ' gelÃ¶schter bereich
        tib[i] := k
        return
      gc#m_c_tag1..gc#m_c_tag10:                        ' colortag gefunden
        tib[i] := k
        return
      TIB_SP:                                           ' space gefunden
        tib[i] := k
        return

PRI tib_setchar(k)                                      'tib: zeichen einfÃ¼gen

      bytemove(@tib+tibpos+1, @tib+tibpos, TIBMAX-tibpos-1)
      tib[tibpos] := k
      if tibpos < TIBMAX-1
        tibpos++

PRI tib_disp(mode) | i,j,colback                        'tib: ausgabe tib

  tib_pos1
  '----------------------------------------------------- infozeile schreiben
  if mode == 1
    colback := color
    color := tibctag - gc#m_c_tag1
    tib_printtag(tibctag)
    print_char($0D)
    color := colback
  '----------------------------------------------------- zeile vor cursor
  i := 0
  if tibpos
    repeat i from 0 to tibpos-1
      print_char(tib[i])
  '----------------------------------------------------- cursor
  colback := color
  color := gc#c_cursor
  if tib[i] > TIB_SP
    print_char(tib[i++]) 'normales zeichen
  else
    print_char(TIB_SP)   'colortag
    i++
  color := colback
  '----------------------------------------------------- zeile nach cursor
  if tibpos < TIBMAX-1
    repeat j from i to TIBMAX-1   'puffer nach cursorposition
      print_char(tib[j])


PRI tib_printtag(tagnr)                                 'tib: ausgabe des tagstrings

  ifnot scr_mode
    case tagnr
      gc#m_c_tag1:   print_str(string(" [execute]        "))
      gc#m_c_tag2:   print_str(string(" [create]         "))
      gc#m_c_tag3:   print_str(string(" [compile]        "))
      gc#m_c_tag4:   print_str(string(" [number]         "))
      gc#m_c_tag5:   print_str(string(" [number-literal] "))
      gc#m_c_tag6:   print_str(string(" [string]         "))
      gc#m_c_tag7:   print_str(string(" [string-literal] "))
      gc#m_c_tag8:   print_str(string(" [data]           "))
      gc#m_c_tag9:   print_str(string(" [remark]         "))
  else
    print_str(string(" [TEXT-MODUS]     "))


PRI tib_print | i                                       'tib: ausgabe tib im logbereich

  repeat i from 0 to TIBMAX-1
    print_char(tib[i])

PRI tib_pos1                                            'tib: cursor auf anfangsposition

  col := 0
  row := tibpos1 ' rows - 6
  eline := rows

PRI tib_copy | i                                        'tib: tib --> kopierpuffer

  repeat i from 0 to TIBMAX+2
    tibcopy[i] := tib[i]
  case scr_mode
    0: tibpos  := 1
    1: tibpos  := 0

PRI tib_past | i                                        'tib: kopierpuffer --> tib

  repeat i from 0 to TIBMAX+2
    tib[i] := tibcopy[i]

PRI tib_delcopy | i                                     'tib: kopierpuffer lÃ¶schen

  repeat i from 0 to TIBMAX+2
    tibcopy[i] := 0
  case scr_mode
    0: tibpos  := 1
    1: tibpos  := 0

PRI tib_fill | i,n                                      'tib: REM --> tib

  case scr_mode
    0: tibpos := 1
       n := gc#m_c_tag9
    0: tibpos := 0
       n := " "

  repeat i from 0 to TIBMAX+2
    tib[i] := n


PRI tib_clear | i                                       'tib: lÃ¶schen

  repeat i from 0 to TIBMAX+2
    tib[i] := 0
  tibpos  := 0
  tibctag := gc#m_c_tag1
  tib_setchar(gc#m_c_tag1)           'default-tag setzen

  tok_tag  := 0
  tok_pos  := 0

PRI tib_error                                           'tib: fehlermeldungen

  color := gc#c_attention
  case error
    gc#M_ERR_RS:  print_str(string(" error returnstack"))
    gc#M_ERR_DS:  print_str(string(" error datastack"))
    gc#M_ERR_IN:  print_char(" ")
             print_str(@tok)
             print_str(string(" ?"))
    gc#M_ERR_CP:  print_char(" ")
             print_str(@tok)
             print_str(string(" ?"))
    gc#M_ERR_SI:  print_str(string(" structure imbalance"))
    gc#M_ERR_SD:  print_str(string(" medium error"))
    gc#M_ERR_RW:  print_str(string(" r/w error"))
    gc#M_ERR_NF:  print_str(string(" not found"))
    gc#M_ERR_ST:  print_str(string(" stack error"))
  color := gc#c_normal

CON ''------------------------------------------------- CHIP-MANAGMENT-FUNKTIONEN

PRI mgr_setcolor|cnr                                    'cmgr: farbregister setzen
''funktionsgruppe               : cmgr
''funktion                      : farbregister setzen
''eingabe                       : -
''ausgabe                       : -
''busprotokoll                  : [0][091][get.cnr][sub_getlong.color]
''                              : cnr   - nummer des farbregisters 0..15
''                              : color - farbwert

  cnr   := bus_getchar
  long[@vgacolors][cnr] := sub_getlong

PRI mgr_load|i                                          'cmgr: bellatrix-loader
''funktionsgruppe               : cmgr
''funktion                      : funktion um einen neuen code in bellatrix zu laden
''
''bekanntes problem: einige wenige bel-dateien werden geladen aber nicht korrekt gestartet
''lÃ¶sung: diese datei als eeprom-image speichern

' kopf der bin-datei einlesen                           ------------------------------------------------------
  repeat i from 0 to 15                                 '16 bytes --> proghdr
    byte[@proghdr][i] := bus_getchar

  plen := 0
  plen :=        byte[@proghdr + $0B] << 8
  plen := plen + byte[@proghdr + $0A]

' objektlÃ¤nge an regnatix senden
  bus_putchar(plen >> 8)                                'hsb senden
  bus_putchar(plen & $FF)                               'lsb senden

  repeat i from 0 to 7                                  'alle anderen cogs anhalten
    ifnot i == cogid
      cogstop(i)

  dira := 0                                             'diese cog vom bus trennen
  cognew(@loader, plen)

  cogstop(cogid)                                        'cog 0 anhalten

DAT                                                     'cmgr: pasm-code loader
                        org     0

loader
                        mov     outa,    M_0               'bus inaktiv
                        mov     dira,    DINP              'bus auf eingabe schalten
                        mov     reg_a,   PAR               'parameter = plen
                        mov     reg_b,   #0                'adresse ab 0

                        ' datenblock empfangen
loop
                        call    #get                       'wert einlesen
                        wrbyte  in,      reg_b             'wert --> hubram
                        add     reg_b,   #1                'adresse + 1
                        djnz    reg_a,   #loop

                        ' neuen code starten

                        rdword  reg_a,   #$A               ' Setup the stack markers.
                        sub     reg_a,   #4                '
                        wrlong  SMARK,   reg_a             '
                        sub     reg_a,   #4                '
                        wrlong  SMARK,   reg_a             '

                        rdbyte  reg_a,   #$4               ' Switch to new clock mode.
                        clkset  reg_a                                             '

                        coginit SINT                       ' Restart running new code.


                        cogid   reg_a
                        cogstop reg_a                      'cog hÃ¤lt sich selbst an


get
                        waitpeq M_1,      M_2              'busclk=1? & /cs=0?
                        mov     in,       ina              'daten einlesen
                        and     in,       DMASK            'wert maskieren
                        mov     outa,     M_3              'hs=0
                        waitpeq M_3,      M_4              'busclk=0?
                        mov     outa,     M_0              'hs=1
get_ret                 ret


'     hbeat   --------+
'     clk     -------+|
'     /wr     ------+||
'     /hs     -----+|||+------------------------- /cs
'                  |||||                 -------- d0..d7
DINP    long  %00001001000000000000000000000000  'constant dinp hex  \ bus input
DOUT    long  %00001001000000000000000011111111  'constant dout hex  \ bus output

M_0     long  %00001000000000000000000000000000  'bus inaktiv

M_1     long  %00000010000000000000000000000000
M_2     long  %00000010100000000000000000000000  'busclk=1? & /cs=0?

M_3     long  %00000000000000000000000000000000
M_4     long  %00000010000000000000000000000000  'busclk=0?


DMASK   long  %00000000000000000000000011111111  'datenmaske

SINT    long    ($0001 << 18) | ($3C01 << 4)                       ' Spin interpreter boot information.
SMARK   long    $FFF9FFFF                                          ' Stack mark used for spin code.

in      res   1
reg_a   res   1
reg_b   res   1




CON ''------------------------------------------------- SCREEN-FUNKTIONEN

PUB print_dec(value,digits) | i                         'screen: dezimalen zahlenwert auf bildschirm ausgeben
{{printdec(value) - screen: dezimale bildschirmausgabe zahlenwertes}}

  if value < 0                                          'negativer zahlenwert
    -value
    print_char("-")
  i := 1_000_000_000
  repeat 10                                             'zahl zerlegen
    if value => i
      print_char(value / i + "0")
      value //= i
      result~~
    elseif result or i == 1
      print_char("0")
    i /= 10                                             'nÃ¤chste stelle
  print_char(" ")

PUB print_hex(value, digits)                            'screen: hexadezimalen zahlenwert auf bildschirm ausgeben
{{hex(value,digits) - screen: hexadezimale bildschirmausgabe eines zahlenwertes}}

  value <<= (8 - digits) << 2
  repeat digits
    print_char(lookupz((value <-= 4) & $F : "0".."9", "A".."F"))
  print_char(" ")

PUB print_bin(value, digits)                            'screen: binÃ¤ren zahlenwert auf bildschirm ausgeben

  value <<= 32 - digits
  repeat digits
    print_char((value <-= 1) & 1 + "0")
  print_char(" ")

PRI print_str(stringptr)                                'screen: bildschirmausgabe einer zeichenkette (0-terminiert)
  repeat strsize(stringptr)
    print_char(byte[stringptr++])

PRI print_char(c) | code,n                              'screen: zeichen auf bildschirm ausgeben
{{zeichen auf bildschirm ausgeben}}

  case c
    $00:                                                'softspace (bella-intern)
      pchar(" ")
    $01:                                                'clear screen?
      n := sline * cols
      wordfill(@array.word[n],spacetile,tiles-n)
      row := sline
      col := 0
    $02:                                                'home?
      row := sline
      col := 0
    $03:                                                'pos1
      col := 0
    $04:                                                'curon
      curstat := 1
    $05:                                                'curoff
      curstat := 0
    $06:                                                'scrollup
      scrollup
    $07:                                                'scrolldown
      scrolldown
    $08:                                                'backspace?
      if col
        col--
    $09:                                                'tab
      repeat n from 0 to TABANZ-1
        if col < tab[n]
          col := tab[n]
          quit
    $0D:                                                'return?
      newline
    '-------------------------------------------------- COLORTAGS
    gc#m_c_tag1:  set_ctag(0)
    gc#m_c_tag2:  set_ctag(1)
    gc#m_c_tag3:  set_ctag(2)
    gc#m_c_tag4:  set_ctag(3)
    gc#m_c_tag5:  set_ctag(4)
    gc#m_c_tag6:  set_ctag(5)
    gc#m_c_tag7:  set_ctag(6)
    gc#m_c_tag8:  set_ctag(7)
    gc#m_c_tag9:  set_ctag(8)
    gc#m_c_tag10: set_ctag(9)
    '-------------------------------------------------- ZEICHEN
    $20..$FF:                                           'character?
      pchar(c)

PRI set_ctag(ctagnr)                                    'screen: colortag ausgeben

    ctag  := ctagnr
    color := ctagnr
    pchar(" ")


PRI pchar(c)| i,k                                       'screen: schreibt zeichen an aktuelle position

  k := color << 1 + c & 1
  i := $8000 + (c & $FE) << 6 + k
  array.word[row * cols + col] := i                                             'oberes tile setzen
  array.word[(row + 1) * cols + col] := i | $40                                 'unteres tile setzen
  if ++col == cols
    newline

PRI newline | i                                         'screen: zeilenwechsel, inkl. scrolling am screenende

  col := 0
  if (row += 2) => eline
    row -= 2
    'scroll lines
    repeat i from sline to eline-3
      wordmove(@array.word[i*cols], @array.word[(i+2)*cols], cols)              'wordmove(dest,src,cnt)
    'clear new line
    wordfill(@array.word[(eline-2)*cols], spacetile, cols<<1)

PRI scrollup | i                                        'screen: scrollt den screen nach oben

    'scroll lines
    wordmove(@array.word[sline*cols],@array.word[(sline+2)*cols],(eline-1-sline)*cols) 'wordmove(dest,src,cnt)
    'clear new line
    wordfill(@array.word[(eline-2)*cols],spacetile,cols<<1)

PRI scrolldown | i                                      'screen: scrollt den screen nach unten
    'scroll lines
    i := eline - 1
    repeat eline-sline-1
      wordmove(@array.word[i*cols],@array.word[(i-2)*cols],cols)              'wordmove(dest,src,cnt)
      i--
    'clear new line
    wordfill(@array.word[(sline)*cols],spacetile,cols<<1)

DAT                                                     'screen: farbpalette

'============================================================
'          v  h  v  h        ' v=Vordergrund, h=Hintergrund
'   long $ 3C 04 3C 04       'Muster
'          v  v  h  h
'   long $ 3C 3C 04 04       'Muster
'0     %%RGBx_RGBx_RGBx_RGBx
' long %%0330_0010_0330_0010
' long %%0330_0330_0010_0010
'============================================================

vgacolors long                                  'farbpalette

'set 1 - m colortags color/black

  long $30003000       'color 0: +grÃ¼n - wort ausfÃ¼hren
  long $30300000

  long $C000C000       'color 1: rot - wort definieren
  long $C0C00000

  long $10001000       'color 2: -grÃ¼n - wort compilieren
  long $10100000

  long $0C000C00       'color 3: +blau - zahl
  long $0C0C0000

  long $04000400       'color 4: -blau - zahl literal
  long $04040000

col_string1
  long $E000E000       'color 6: +gelb - string
col_string2
  long $E0E00000

col_stringlit1
  long $50005000       'color 6: -gelb - string literal
col_stringlit2
  long $50500000

  long $C800C800       'color 7: magenta - data
  long $C8C80000

  long $54005400       'color 8: grau - kommentar
  long $54540000

  long $54005400       'color 9: grau - kommentar
  long $54540000

  long $54005400       'color 10: grau - kommentar
  long $54540000

  long $54005400       'color 11: grau - kommentar
  long $54540000

  long $54005400       'color 12: grau - kommentar
  long $54540000

  long $54005400       'color 13: grau - kommentar
  long $54540000

  long $54005400       'color 14: grau - kommentar
  long $54540000

cursordat1
  long $30003000       'color 15: invers cursor
cursordat2
  long $30300000



'set 2 - m colortags color/white
{
  long $00FC00FC       'color 0: schwarz - wort ausfÃ¼hren
  long $0000FCFC

  long $C0FCC0FC       'color 1: rot - wort definieren
  long $C0C0FCFC

  long $30FC30FC       'color 2: grÃ¼n - wort compilieren
  long $3030FCFC

  long $0CFC0CFC       'color 3: blau - zahl
  long $0C0CFCFC

  long $3CFC3CFC       'color 4: cyan - zahl literal
  long $3C3CFCFC
col_string1
  long $D0FCD0FC       'color 5: orange - string
col_string2
  long $D0D0FCFC
col_stringlit1
  long $E0FCE0FC       'color 6: gelb - string literal
col_stringlit2
  long $E0E0FCFC

  long $C8FCC8FC       'color 7: magenta - data
  long $C8C8FCFC
col_remark1
  long $54FC54FC       'color 8: grau - kommentar
col_remark2
  long $5454FCFC

  long $54FC54FC       'color 9: grau - kommentar
  long $5454FCFC

  long $54FC54FC       'color 10: grau - kommentar
  long $5454FCFC

  long $54FC54FC       'color 11: grau - kommentar
  long $5454FCFC

  long $54FC54FC       'color 12: grau - kommentar
  long $5454FCFC

  long $54FC54FC       'color 13: grau - kommentar
  long $5454FCFC

  long $54FC54FC       'color 14: grau - kommentar
  long $5454FCFC

cursordat1
  long $FC00FC00       'color 15: invers cursor
cursordat2
  long $FCFC0000
}

DAT                                                     'lizenz

{{

┌──────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
│                                                   TERMS OF USE: MIT License                                                  │
├──────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
│Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    │
│files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    │
│modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software│
│is furnished to do so, subject to the following conditions:                                                                   │
│                                                                                                                              │
│The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.│
│                                                                                                                              │
│THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          │
│WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         │
│COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   │
│ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         │
└──────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
}}
